import './bootstrap';

// CSRF Token Auto-Refresh & Session Keep-Alive
// Keeps the session alive and prevents "Page Expired" errors
(function() {
    // Refresh CSRF token every 10 minutes
    const REFRESH_INTERVAL = 10 * 60 * 1000; // 10 minutes
    
    async function refreshCsrfToken() {
        try {
            const response = await fetch('/csrf-refresh', {
                method: 'GET',
                credentials: 'same-origin',
                headers: {
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                const newToken = data.token;
                
                if (newToken) {
                    // Update the CSRF token in the meta tag
                    const metaTag = document.querySelector('meta[name="csrf-token"]');
                    if (metaTag) {
                        metaTag.setAttribute('content', newToken);
                    }
                    
                    // Update all hidden CSRF input fields
                    document.querySelectorAll('input[name="_token"]').forEach(input => {
                        input.value = newToken;
                    });
                    
                    // Update axios default header
                    if (window.axios) {
                        window.axios.defaults.headers.common['X-CSRF-TOKEN'] = newToken;
                    }
                    
                    console.debug('CSRF token refreshed');
                }
            }
        } catch (error) {
            console.warn('Failed to refresh CSRF token:', error);
        }
    }
    
    // Refresh token periodically
    setInterval(refreshCsrfToken, REFRESH_INTERVAL);
    
    // Also refresh when tab becomes visible (user returns to tab)
    document.addEventListener('visibilitychange', function() {
        if (document.visibilityState === 'visible') {
            // Small delay to avoid rapid-fire requests
            setTimeout(refreshCsrfToken, 500);
        }
    });
    
    // Refresh on window focus (backup for visibility change)
    window.addEventListener('focus', function() {
        setTimeout(refreshCsrfToken, 500);
    });
})();

// Livewire Page Expired Handler
// Automatically refreshes the page on 419 errors instead of showing an alert
document.addEventListener('livewire:init', () => {
    Livewire.hook('request', ({ fail }) => {
        fail(({ status, preventDefault }) => {
            if (status === 419) {
                // Page expired - refresh the page silently
                preventDefault();
                window.location.reload();
            }
        });
    });
});
