# CBT System - Shared Hosting Deployment Guide

This guide walks you through deploying the CBT System to a shared hosting environment (e.g., cPanel, DirectAdmin, Plesk).

## Prerequisites

Ensure your shared hosting provides:
- **PHP 8.2+** (preferably 8.3 or 8.4)
- **MySQL 8.0+** or **PostgreSQL 13+**
- **Composer** (or SSH access to install dependencies)
- **Node.js/NPM** (for building assets, or build locally before upload)
- PHP Extensions: `BCMath`, `Ctype`, `Fileinfo`, `JSON`, `Mbstring`, `OpenSSL`, `PDO`, `Tokenizer`, `XML`, `cURL`, `GD` or `Imagick`

---

## Step 1: Prepare the Application Locally

### 1.1 Build Frontend Assets
```bash
npm install
npm run build
```

### 1.2 Install Production Dependencies
```bash
composer install --optimize-autoloader --no-dev
```

### 1.3 Clear Development Caches
```bash
php artisan optimize:clear
```

---

## Step 2: Prepare Files for Upload

### Files/Folders to Upload
Upload **everything** except:
- `node_modules/` (not needed in production)
- `.git/` (version control)
- `tests/` (optional, not needed in production)
- `docker/`, `docker-compose.yml`, `Dockerfile` (Docker files)
- `.env` (create new one on server)
- `storage/logs/*` (create empty)
- `storage/framework/cache/*`, `storage/framework/sessions/*`, `storage/framework/views/*` (create empty directories)

### Recommended Upload Structure
```
your-domain.com/
├── public_html/          ← Contents of /public folder go here
│   ├── index.php
│   ├── .htaccess
│   ├── build/
│   ├── css/
│   ├── js/
│   ├── fonts/
│   ├── images/
│   └── storage → ../cbt-app/storage/app/public (symlink)
│
└── cbt-app/              ← Main application (OUTSIDE public_html)
    ├── app/
    ├── bootstrap/
    ├── config/
    ├── database/
    ├── resources/
    ├── routes/
    ├── storage/
    ├── vendor/
    ├── .env
    ├── artisan
    └── composer.json
```

---

## Step 3: Upload Files

### Option A: Using File Manager (cPanel)
1. Create a folder **outside** `public_html` called `cbt-app`
2. Upload all application files to `cbt-app/`
3. Upload contents of `public/` folder to `public_html/`

### Option B: Using SSH/Terminal
```bash
# Connect via SSH
ssh username@your-server.com

# Navigate to home directory
cd ~

# Upload via SCP or SFTP, then extract if zipped
unzip cbt-system.zip -d cbt-app
mv cbt-app/public/* public_html/
```

### Option C: Using Git (if SSH available)
```bash
cd ~
git clone https://github.com/your-repo/cbt-system.git cbt-app
cd cbt-app
composer install --optimize-autoloader --no-dev
```

---

## Step 4: Configure public/index.php

Edit `public_html/index.php` to point to the application folder:

```php
<?php

use Illuminate\Http\Request;

define('LARAVEL_START', microtime(true));

// Determine if the application is in maintenance mode...
if (file_exists($maintenance = __DIR__.'/../cbt-app/storage/framework/maintenance.php')) {
    require $maintenance;
}

// Register the Composer autoloader...
require __DIR__.'/../cbt-app/vendor/autoload.php';

// Bootstrap Laravel and handle the request...
(require_once __DIR__.'/../cbt-app/bootstrap/app.php')
    ->handleRequest(Request::capture());
```

---

## Step 5: Create Database

### Using cPanel:
1. Go to **MySQL Databases**
2. Create a new database (e.g., `username_cbt`)
3. Create a new user with a strong password
4. Add user to database with **ALL PRIVILEGES**

### Using phpMyAdmin:
1. Create database with `utf8mb4_unicode_ci` collation

---

## Step 6: Configure Environment (.env)

Create `.env` file in `cbt-app/` folder:

```env
APP_NAME="CBT System - FNPH"
APP_ENV=production
APP_KEY=
APP_DEBUG=false
APP_URL=https://your-domain.com

APP_LOCALE=en
APP_FALLBACK_LOCALE=en
APP_FAKER_LOCALE=en_US

APP_MAINTENANCE_DRIVER=file

BCRYPT_ROUNDS=12

LOG_CHANNEL=stack
LOG_STACK=single
LOG_DEPRECATIONS_CHANNEL=null
LOG_LEVEL=error

# Database Configuration (MySQL example)
DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=username_cbt
DB_USERNAME=username_cbt_user
DB_PASSWORD=your_secure_password

# Session & Cache (use database for shared hosting)
SESSION_DRIVER=database
SESSION_LIFETIME=1440
SESSION_ENCRYPT=true
SESSION_PATH=/
SESSION_DOMAIN=null

BROADCAST_CONNECTION=log
FILESYSTEM_DISK=local
QUEUE_CONNECTION=database

CACHE_STORE=database

# Mail Configuration (update with your SMTP details)
MAIL_MAILER=smtp
MAIL_HOST=mail.your-domain.com
MAIL_PORT=465
MAIL_USERNAME=noreply@your-domain.com
MAIL_PASSWORD=your_email_password
MAIL_ENCRYPTION=ssl
MAIL_FROM_ADDRESS="noreply@your-domain.com"
MAIL_FROM_NAME="${APP_NAME}"

VITE_APP_NAME="${APP_NAME}"
```

---

## Step 7: Set Directory Permissions

Via SSH or File Manager, set permissions:

```bash
cd ~/cbt-app

# Storage and cache directories need to be writable
chmod -R 775 storage
chmod -R 775 bootstrap/cache

# Ensure the web server can write
chown -R username:www-data storage
chown -R username:www-data bootstrap/cache
```

Or via cPanel File Manager:
- Right-click on `storage/` → Change Permissions → `775` (recursive)
- Right-click on `bootstrap/cache/` → Change Permissions → `775` (recursive)

---

## Step 8: Run Artisan Commands

### Via SSH (Recommended):
```bash
cd ~/cbt-app

# Generate application key
php artisan key:generate

# Run database migrations
php artisan migrate --force

# Run REQUIRED seeders (roles and permissions only)
php artisan db:seed --class=RolesAndPermissionsSeeder

# Create storage symlink
php artisan storage:link

# Cache configuration for performance
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### Via cPanel Cron Jobs (Alternative):
If no SSH access, create temporary cron jobs that run once:

```bash
cd ~/cbt-app && /usr/local/bin/php artisan key:generate
cd ~/cbt-app && /usr/local/bin/php artisan migrate --force
cd ~/cbt-app && /usr/local/bin/php artisan db:seed --class=RolesAndPermissionsSeeder
cd ~/cbt-app && /usr/local/bin/php artisan storage:link
```

Run each, then delete the cron job.

---

## Step 9: Create Storage Symlink Manually (if needed)

If `php artisan storage:link` doesn't work, create manually:

### Via SSH:
```bash
cd ~/public_html
ln -s ../cbt-app/storage/app/public storage
```

### Via cPanel File Manager:
This is trickier. You may need to:
1. Create a redirect in `.htaccess` for storage URLs, OR
2. Contact your hosting support to create the symlink

---

## Step 10: Configure .htaccess

Ensure `public_html/.htaccess` contains:

```apache
<IfModule mod_rewrite.c>
    <IfModule mod_negotiation.c>
        Options -MultiViews -Indexes
    </IfModule>

    RewriteEngine On

    # Handle Authorization Header
    RewriteCond %{HTTP:Authorization} .
    RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

    # Redirect Trailing Slashes If Not A Folder...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]

    # Send Requests To Front Controller...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ index.php [L]
</IfModule>
```

---

## Step 11: Create First Admin User

After deployment, you need to create the first administrator account.

### Via Tinker (SSH):
```bash
cd ~/cbt-app
php artisan tinker
```

Then run:
```php
use App\Models\Staff;
use Illuminate\Support\Facades\Hash;

$admin = Staff::create([
    'staff_number' => 'PN001',
    'full_name' => 'Admin User',
    'email' => 'admin@your-domain.com',
    'password' => Hash::make('YourSecurePassword123!'),
    'staff_type' => 'senior',
    'employment_status' => 'active',
]);

$admin->assignRole('Examiner');

exit
```

---

## Step 12: SSL Certificate

Enable HTTPS for your domain:
1. Go to cPanel → **SSL/TLS Status** or **Let's Encrypt**
2. Issue a free Let's Encrypt certificate
3. Force HTTPS redirect (add to `.htaccess`):

```apache
# Force HTTPS
RewriteCond %{HTTPS} off
RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
```

---

## Step 13: Set Up Queue Worker (Important!)

The CBT system uses queues for:
- Question imports
- Email sending
- Background processing

### Option A: Cron-based Queue (Recommended for Shared Hosting)
Add to cPanel Cron Jobs (run every minute):
```bash
cd ~/cbt-app && /usr/local/bin/php artisan queue:work --stop-when-empty --max-time=55
```

### Option B: Schedule Runner
Add to cPanel Cron Jobs (run every minute):
```bash
cd ~/cbt-app && /usr/local/bin/php artisan schedule:run >> /dev/null 2>&1
```

---

## Seeders Reference

### REQUIRED for Production:
```bash
php artisan db:seed --class=RolesAndPermissionsSeeder
```
This creates the Examiner role with full system permissions.

### DO NOT RUN in Production:
- `DatabaseSeeder` - Creates test user
- `TestDataSeeder` - Creates test data (students, exams, questions)
- `EssayQuestionsSeeder` - Creates sample essay questions

---

## Post-Deployment Checklist

- [ ] Application loads without errors
- [ ] Admin login works (`/admin/login`)
- [ ] Can create staff accounts
- [ ] Can create students
- [ ] Can create question categories
- [ ] Can create questions
- [ ] Can create exams
- [ ] Can create exam sessions
- [ ] Student login works (`/`)
- [ ] File uploads work (logo, favicon, background image)
- [ ] Queue processing works (test question import)
- [ ] Emails send correctly (if configured)

---

## Troubleshooting

### 500 Internal Server Error
1. Check `storage/logs/laravel.log` for errors
2. Verify directory permissions (775 on storage/)
3. Ensure `.env` file exists and has correct database credentials

### Blank White Page
1. Enable debug temporarily: `APP_DEBUG=true`
2. Check PHP error logs in cPanel
3. Verify PHP version compatibility

### "Class not found" Errors
```bash
cd ~/cbt-app
composer dump-autoload
php artisan config:clear
```

### Session/Login Issues
1. Verify session table exists: `php artisan migrate`
2. Clear session cache: `php artisan cache:clear`
3. Check `SESSION_DOMAIN` in `.env`

### Storage/Upload Issues
1. Verify storage symlink exists
2. Check permissions on `storage/app/public/`
3. Ensure `FILESYSTEM_DISK=local` in `.env`

---

## Security Recommendations

1. **Remove .env.example** after deployment
2. **Disable directory listing** in .htaccess
3. **Keep APP_DEBUG=false** in production
4. **Use strong passwords** for database and admin accounts
5. **Enable SSL** and force HTTPS
6. **Regular backups** of database and storage folder
7. **Keep PHP and dependencies updated**

---

## Support

If you encounter issues:
1. Check Laravel logs: `storage/logs/laravel.log`
2. Check PHP error logs in cPanel
3. Verify all environment variables are set correctly
