<?php

namespace App\Settings;

use Spatie\LaravelSettings\Settings;

class FeatureSettings extends Settings
{
    public bool $secondary_user_exams;
    public bool $practice_mode;
    public bool $certificate_generation;
    public bool $result_printing;
    public bool $bulk_import;
    public bool $api_access;
    public bool $dark_mode;

    public static function group(): string
    {
        return 'features';
    }

    /**
     * Check if a specific feature is enabled.
     */
    public function isEnabled(string $feature): bool
    {
        $property = $this->normalizeFeatureName($feature);

        if (property_exists($this, $property)) {
            return (bool) $this->{$property};
        }

        return false;
    }

    /**
     * Get all enabled features as an array.
     */
    public function getEnabledFeatures(): array
    {
        return array_filter([
            'secondary_user_exams' => $this->secondary_user_exams,
            'practice_mode' => $this->practice_mode,
            'certificate_generation' => $this->certificate_generation,
            'result_printing' => $this->result_printing,
            'bulk_import' => $this->bulk_import,
            'api_access' => $this->api_access,
            'dark_mode' => $this->dark_mode,
        ]);
    }

    /**
     * Get feature descriptions for UI display.
     */
    public static function getFeatureDescriptions(): array
    {
        return [
            'secondary_user_exams' => [
                'label' => 'Staff/Secondary User Exams',
                'description' => 'Allow secondary users (staff, examiners) to take exams in addition to primary users.',
                'icon' => 'heroicon-o-user-group',
            ],
            'practice_mode' => [
                'label' => 'Practice Mode',
                'description' => 'Allow users to take practice exams without affecting their official scores.',
                'icon' => 'heroicon-o-academic-cap',
            ],
            'certificate_generation' => [
                'label' => 'Certificate Generation',
                'description' => 'Automatically generate certificates for users who pass exams.',
                'icon' => 'heroicon-o-document-check',
            ],
            'result_printing' => [
                'label' => 'Result Printing',
                'description' => 'Allow users to print their exam results.',
                'icon' => 'heroicon-o-printer',
            ],
            'bulk_import' => [
                'label' => 'Bulk Import',
                'description' => 'Import questions and users from Excel/CSV files.',
                'icon' => 'heroicon-o-arrow-up-tray',
            ],
            'api_access' => [
                'label' => 'API Access',
                'description' => 'Enable REST API for external integrations.',
                'icon' => 'heroicon-o-code-bracket',
            ],
            'dark_mode' => [
                'label' => 'Dark Mode',
                'description' => 'Allow users to switch to dark mode interface.',
                'icon' => 'heroicon-o-moon',
            ],
        ];
    }

    /**
     * Normalize feature name to property name.
     */
    protected function normalizeFeatureName(string $feature): string
    {
        return str_replace(['-', '.'], '_', strtolower($feature));
    }
}
