<?php

namespace App\Settings;

use Illuminate\Support\Facades\Storage;
use Spatie\LaravelSettings\Settings;

class BrandingSettings extends Settings
{
    public ?string $logo_path;
    public ?string $favicon_path;
    public ?string $auth_background_image;
    public string $primary_color;
    public string $secondary_color;
    public string $accent_color;

    public static function group(): string
    {
        return 'branding';
    }

    /**
     * Get the full URL to the logo.
     */
    public function getLogoUrl(): ?string
    {
        if (! $this->logo_path) {
            return null;
        }

        return Storage::disk('public')->url($this->logo_path);
    }

    /**
     * Get the full URL to the favicon.
     */
    public function getFaviconUrl(): ?string
    {
        if (! $this->favicon_path) {
            return null;
        }

        return Storage::disk('public')->url($this->favicon_path);
    }

    /**
     * Get the full URL to the auth background image.
     */
    public function getAuthBackgroundUrl(): ?string
    {
        if (! $this->auth_background_image) {
            return asset('images/fnph-building.jpg'); // Default fallback
        }

        return Storage::disk('public')->url($this->auth_background_image);
    }

    /**
     * Get CSS variables for theming.
     */
    public function getCssVariables(): array
    {
        return [
            '--color-primary' => $this->primary_color,
            '--color-secondary' => $this->secondary_color,
            '--color-accent' => $this->accent_color,
        ];
    }

    /**
     * Generate inline CSS for theme colors.
     */
    public function getInlineCss(): string
    {
        $vars = $this->getCssVariables();
        $css = ':root {';

        foreach ($vars as $property => $value) {
            $css .= "{$property}: {$value};";
        }

        $css .= '}';

        return $css;
    }

    /**
     * Convert hex color to Filament-compatible RGB array.
     */
    public function hexToRgb(string $hex): array
    {
        $hex = ltrim($hex, '#');

        return [
            'r' => hexdec(substr($hex, 0, 2)),
            'g' => hexdec(substr($hex, 2, 2)),
            'b' => hexdec(substr($hex, 4, 2)),
        ];
    }

    /**
     * Get Filament-compatible color array for primary color.
     */
    public function getFilamentPrimaryColor(): array
    {
        return $this->generateColorPalette($this->primary_color);
    }

    /**
     * Get Filament-compatible color array for secondary color.
     */
    public function getFilamentSecondaryColor(): array
    {
        return $this->generateColorPalette($this->secondary_color);
    }

    /**
     * Get Filament-compatible color array for accent color.
     */
    public function getFilamentAccentColor(): array
    {
        return $this->generateColorPalette($this->accent_color);
    }

    /**
     * Generate a full color palette from a base hex color.
     * This creates shades from 50 to 950 for Tailwind/Filament compatibility.
     */
    public function generateColorPalette(string $baseHex): array
    {
        $rgb = $this->hexToRgb($baseHex);

        // Generate palette with proper shade distribution
        return [
            50 => $this->adjustBrightness($rgb, 0.95),
            100 => $this->adjustBrightness($rgb, 0.9),
            200 => $this->adjustBrightness($rgb, 0.75),
            300 => $this->adjustBrightness($rgb, 0.6),
            400 => $this->adjustBrightness($rgb, 0.3),
            500 => "rgb({$rgb['r']}, {$rgb['g']}, {$rgb['b']})",
            600 => $this->adjustBrightness($rgb, -0.1),
            700 => $this->adjustBrightness($rgb, -0.25),
            800 => $this->adjustBrightness($rgb, -0.4),
            900 => $this->adjustBrightness($rgb, -0.55),
            950 => $this->adjustBrightness($rgb, -0.7),
        ];
    }

    /**
     * Adjust brightness of RGB color.
     */
    protected function adjustBrightness(array $rgb, float $factor): string
    {
        if ($factor > 0) {
            // Lighten
            $r = $rgb['r'] + (255 - $rgb['r']) * $factor;
            $g = $rgb['g'] + (255 - $rgb['g']) * $factor;
            $b = $rgb['b'] + (255 - $rgb['b']) * $factor;
        } else {
            // Darken
            $r = $rgb['r'] * (1 + $factor);
            $g = $rgb['g'] * (1 + $factor);
            $b = $rgb['b'] * (1 + $factor);
        }

        return sprintf('rgb(%d, %d, %d)', (int) $r, (int) $g, (int) $b);
    }
}
