<?php

namespace App\Services;

use App\Models\QuestionBank;
use App\Jobs\ProcessQuestionImport;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpWord\IOFactory;
use PhpOffice\PhpWord\Element\Text;
use PhpOffice\PhpWord\Element\TextRun;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class QuestionImportService
{
    /**
     * Process uploaded file for question import
     */
    public function processImport(UploadedFile $file, array $metadata): array
    {
        $extension = $file->getClientOriginalExtension();
        $batchId = (string) Str::uuid();
        
        try {
            $questions = match (strtolower($extension)) {
                'xlsx', 'xls', 'csv' => $this->parseExcelFile($file),
                'docx', 'doc' => $this->parseWordDocument($file),
                default => throw new \InvalidArgumentException("Unsupported file type: {$extension}")
            };

            // Add metadata to all questions
            $questions = $this->addMetadataToQuestions($questions, $metadata, $batchId);

            // Process all imports synchronously for immediate feedback
            $imported = $this->importQuestions($questions);
            return [
                'success' => true,
                'message' => "Successfully imported {$imported} questions.",
                'batch_id' => $batchId,
                'count' => $imported,
                'processing' => false
            ];
        } catch (\Exception $e) {
            Log::error('Question import failed', [
                'file' => $file->getClientOriginalName(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => 'Import failed: ' . $e->getMessage(),
                'batch_id' => null,
                'count' => 0,
                'processing' => false
            ];
        }
    }

    /**
     * Parse Excel/CSV file
     */
    private function parseExcelFile(UploadedFile $file): array
    {
        // For small files, use the robust import class
        $tempBatchId = (string) Str::uuid();
        $import = new \App\Imports\QuestionsImport(
            ['target_audience' => 'student', 'session' => null, 'department' => null],
            $tempBatchId
        );
        
        try {
            $collection = Excel::toCollection($import, $file);
            $questions = [];
            
            foreach ($collection[0] as $row) {
                if (empty($row['question'])) continue;
                
                $question = $this->parseRowToQuestion($row->toArray(), array_keys($row->toArray()));
                if ($question) {
                    $questions[] = $question;
                }
            }
            
            return $questions;
        } catch (\Exception $e) {
            // Fallback to manual parsing
            $data = Excel::toArray(new class implements \Maatwebsite\Excel\Concerns\ToArray {
                public function array(array $array)
                {
                    return $array;
                }
            }, $file)[0]; // Get first sheet

            $questions = [];
            $headers = array_shift($data); // Remove header row
            
            foreach ($data as $row) {
                if (empty($row[0])) continue; // Skip empty rows
                
                $question = $this->parseRowToQuestion($row, $headers);
                if ($question) {
                    $questions[] = $question;
                }
            }

            return $questions;
        }
    }

    /**
     * Parse Word document
     */
    private function parseWordDocument(UploadedFile $file): array
    {
        $phpWord = IOFactory::load($file->getPathname());
        $questions = [];
        $currentQuestion = null;
        $currentOptions = [];
        $currentAnswer = null;
        $currentExplanation = null;
        
        foreach ($phpWord->getSections() as $section) {
            foreach ($section->getElements() as $element) {
                $text = $this->extractTextFromElement($element);
                
                if (empty($text)) continue;
                
                // Split text by newlines to handle multiple lines in a single element
                // This handles Word documents where soft returns are used instead of paragraphs
                $lines = preg_split('/[\r\n]+/', $text);
                
                foreach ($lines as $line) {
                    $line = trim($line);
                    if (empty($line)) continue;
                    
                    // Detect question patterns
                    if ($this->isQuestionStart($line)) {
                        // Save previous question if exists
                        if ($currentQuestion) {
                            $questions[] = $this->buildQuestionFromWordData($currentQuestion, $currentOptions, $currentAnswer, $currentExplanation);
                        }
                        
                        $currentQuestion = $line;
                        $currentOptions = [];
                        $currentAnswer = null;
                        $currentExplanation = null;
                    } elseif ($this->isAnswerLine($line)) {
                        // Capture answer lines like "Answer: True *" or "Answer: False"
                        $currentAnswer = $line;
                    } elseif ($this->isExplanationLine($line)) {
                        // Capture explanation lines
                        $currentExplanation = $line;
                    } elseif ($this->isOption($line)) {
                        $currentOptions[] = $line;
                    }
                }
            }
        }
        
        // Don't forget the last question
        if ($currentQuestion) {
            $questions[] = $this->buildQuestionFromWordData($currentQuestion, $currentOptions, $currentAnswer, $currentExplanation);
        }

        return $questions;
    }

    /**
     * Extract text from Word element
     */
    private function extractTextFromElement($element): string
    {
        if ($element instanceof Text) {
            return $element->getText();
        } elseif ($element instanceof TextRun) {
            $text = '';
            foreach ($element->getElements() as $textElement) {
                if ($textElement instanceof Text) {
                    $text .= $textElement->getText();
                }
            }
            return $text;
        }
        
        return '';
    }

    /**
     * Check if text starts a new question
     */
    private function isQuestionStart(string $text): bool
    {
        return preg_match('/^\d+[\.\)]\s*(.+)/i', $text) || 
               preg_match('/^Q\d+[:.]?\s*(.+)/i', $text) ||
               preg_match('/^Question\s*\d+[:.]?\s*(.+)/i', $text);
    }

    /**
     * Check if text is an option
     */
    private function isOption(string $text): bool
    {
        return preg_match('/^[a-d][\.\)]\s*(.+)/i', $text) ||
               preg_match('/^\([a-d]\)\s*(.+)/i', $text);
    }

    /**
     * Check if text is an answer line (for True/False questions)
     */
    private function isAnswerLine(string $text): bool
    {
        return preg_match('/^(answer|ans)[\s:]+.*(true|false)/i', $text);
    }

    /**
     * Check if text is an explanation line
     */
    private function isExplanationLine(string $text): bool
    {
        return preg_match('/^explanation[\s:]+/i', $text);
    }

    /**
     * Build question from Word document data
     */
    private function buildQuestionFromWordData(string $questionText, array $options, ?string $answerLine = null, ?string $explanationLine = null): array
    {
        // Clean question text
        $questionText = preg_replace('/^\d+[\.\)]\s*/', '', $questionText);
        $questionText = preg_replace('/^Q\d+[:.]?\s*/', '', $questionText);
        $questionText = preg_replace('/^Question\s*\d+[:.]?\s*/', '', $questionText);
        
        // Determine question type
        $questionType = $this->detectQuestionType($questionText, $options, $answerLine);
        
        // Parse options
        $parsedOptions = [];
        $correctAnswer = null;
        $explanation = null;
        
        // Extract explanation if present
        if ($explanationLine) {
            $explanation = trim(preg_replace('/^explanation[\s:]+/i', '', $explanationLine));
        }
        
        if ($questionType === 'true_false') {
            $parsedOptions = ['true' => 'True', 'false' => 'False'];
            $correctAnswer = $this->extractTrueFalseAnswer($questionText, $answerLine);
            
            // Clean up question text - remove the (True/False) marker
            $questionText = preg_replace('/\s*\(true[\/]?false\)\s*/i', '', $questionText);
        } else {
            foreach ($options as $option) {
                $optionData = $this->parseOption($option);
                if ($optionData) {
                    $parsedOptions[$optionData['key']] = $optionData['text'];
                    if ($optionData['is_correct']) {
                        $correctAnswer = $optionData['key'];
                    }
                }
            }
        }

        return [
            'question_text' => trim($questionText),
            'question_type' => $questionType,
            'options' => $parsedOptions,
            'correct_answer' => $correctAnswer ?? 'a',
            'explanation' => $explanation,
            'time_limit' => null,
            'is_active' => true
        ];
    }

    /**
     * Detect question type from content
     */
    private function detectQuestionType(string $questionText, array $options, ?string $answerLine = null): string
    {
        $lowerText = strtolower($questionText);
        
        // Check for True/False indicators in question text
        if (str_contains($lowerText, 'true or false') || 
            str_contains($lowerText, 'true/false') ||
            str_contains($lowerText, '(true/false)') ||
            str_contains($lowerText, 't/f')) {
            return 'true_false';
        }
        
        // Check if we have an answer line that indicates True/False
        if ($answerLine && preg_match('/^(answer|ans)[\s:]+.*(true|false)/i', $answerLine)) {
            return 'true_false';
        }
        
        // No options usually means True/False
        if (count($options) === 0 && $answerLine) {
            return 'true_false';
        }
        
        return 'single_select';
    }

    /**
     * Extract True/False answer from question text or answer line
     */
    private function extractTrueFalseAnswer(string $questionText, ?string $answerLine): string
    {
        // First check the answer line
        if ($answerLine) {
            $lowerAnswer = strtolower($answerLine);
            if (str_contains($lowerAnswer, 'true')) {
                return 'true';
            } elseif (str_contains($lowerAnswer, 'false')) {
                return 'false';
            }
        }
        
        // Then check question text for embedded answer
        $lowerText = strtolower($questionText);
        if (str_contains($lowerText, 'answer: true') || str_contains($lowerText, 'ans: true')) {
            return 'true';
        } elseif (str_contains($lowerText, 'answer: false') || str_contains($lowerText, 'ans: false')) {
            return 'false';
        }
        
        return 'true'; // Default
    }

    /**
     * Parse individual option
     */
    private function parseOption(string $option): ?array
    {
        $option = trim($option);
        
        // Check for correct answer indicators
        $isCorrect = str_contains($option, '*') || str_contains($option, '(correct)');
        
        // Extract option key and text
        if (preg_match('/^([a-d])[\.\)]\s*(.+)/i', $option, $matches)) {
            return [
                'key' => strtolower($matches[1]),
                'text' => trim(str_replace(['*', '(correct)'], '', $matches[2])),
                'is_correct' => $isCorrect
            ];
        }
        
        if (preg_match('/^\(([a-d])\)\s*(.+)/i', $option, $matches)) {
            return [
                'key' => strtolower($matches[1]),
                'text' => trim(str_replace(['*', '(correct)'], '', $matches[2])),
                'is_correct' => $isCorrect
            ];
        }
        
        return null;
    }

    /**
     * Parse Excel row to question
     */
    private function parseRowToQuestion(array $row, array $headers): ?array
    {
        // Map headers to expected format
        $headerMap = [
            'question' => ['question', 'question_text', 'question text'],
            'type' => ['type', 'question_type', 'question type'],
            'option_a' => ['option_a', 'option a', 'a'],
            'option_b' => ['option_b', 'option b', 'b'],
            'option_c' => ['option_c', 'option c', 'c'],
            'option_d' => ['option_d', 'option d', 'd'],
            'correct_answer' => ['correct_answer', 'correct answer', 'answer'],
            'explanation' => ['explanation', 'explanation'],
        ];
        
        $questionData = [];
        
        // Check if the row is already an associative array (from WithHeadingRow)
        $isAssociative = array_keys($row) !== range(0, count($row) - 1);
        
        foreach ($headerMap as $field => $possibleHeaders) {
            foreach ($possibleHeaders as $header) {
                if ($isAssociative) {
                    // For associative arrays, check if the key exists directly
                    $lowerHeader = strtolower($header);
                    if (isset($row[$lowerHeader])) {
                        $questionData[$field] = $row[$lowerHeader];
                        break;
                    }
                } else {
                    // For numeric arrays, find the index
                    $index = array_search(strtolower($header), array_map('strtolower', $headers));
                    if ($index !== false && isset($row[$index])) {
                        $questionData[$field] = $row[$index];
                        break;
                    }
                }
            }
        }
        
        if (empty($questionData['question'])) {
            return null;
        }
        
        // Build options array
        $options = [];
        $questionType = strtolower($questionData['type'] ?? 'single_select');
        
        if ($questionType === 'true_false') {
            $options = ['true' => 'True', 'false' => 'False'];
        } else {
            foreach (['a', 'b', 'c', 'd'] as $key) {
                if (!empty($questionData["option_{$key}"])) {
                    $options[$key] = $questionData["option_{$key}"];
                }
            }
        }
        
        return [
            'question_text' => $questionData['question'],
            'question_type' => $questionType === 'true_false' ? 'true_false' : 'single_select',
            'options' => $options,
            'correct_answer' => $questionData['correct_answer'] ?? 'a',
            'explanation' => $questionData['explanation'] ?? null,
            'time_limit' => null,
            'is_active' => true
        ];
    }

    /**
     * Add metadata to questions
     */
    private function addMetadataToQuestions(array $questions, array $metadata, string $batchId): array
    {
        $now = now();
        
        return array_map(function ($question) use ($metadata, $batchId, $now) {
            return array_merge($question, [
                'target_audience' => $metadata['target_audience'],
                'session' => $metadata['session'] ?? null,
                'category_id' => $metadata['category_id'] ?? null,
                'department' => $metadata['department'] ?? null,
                'import_batch_id' => $batchId,
                'imported_at' => $now,
                'created_by' => auth()->id(),
                'updated_by' => auth()->id(),
                'created_at' => $now,
                'updated_at' => $now,
            ]);
        }, $questions);
    }

    /**
     * Import questions to database
     */
    public function importQuestions(array $questions): int
    {
        $imported = 0;
        
        DB::transaction(function () use ($questions, &$imported) {
            foreach (array_chunk($questions, 100) as $chunk) {
                // Validate each question
                $validQuestions = array_filter($chunk, [$this, 'validateQuestion']);
                
                if (!empty($validQuestions)) {
                    // Prepare questions for raw insert (encode arrays to JSON)
                    $preparedQuestions = array_map(function ($question) {
                        // Ensure options is JSON encoded for raw insert
                        if (isset($question['options']) && is_array($question['options'])) {
                            $question['options'] = json_encode($question['options']);
                        }
                        // Format datetime for raw insert
                        foreach (['imported_at', 'created_at', 'updated_at'] as $dateField) {
                            if (isset($question[$dateField]) && $question[$dateField] instanceof \DateTimeInterface) {
                                $question[$dateField] = $question[$dateField]->format('Y-m-d H:i:s');
                            }
                        }
                        return $question;
                    }, $validQuestions);
                    
                    QuestionBank::insert($preparedQuestions);
                    $imported += count($preparedQuestions);
                }
            }
        });
        
        return $imported;
    }

    /**
     * Validate question data
     */
    public function validateQuestion(array $question): bool
    {
        return !empty($question['question_text']) &&
               in_array($question['question_type'], QuestionBank::QUESTION_TYPES) &&
               in_array($question['target_audience'], QuestionBank::TARGET_AUDIENCES) &&
               !empty($question['correct_answer']) &&
               is_array($question['options']) &&
               !empty($question['options']);
    }
}