<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;

class Staff extends Authenticatable
{
    use HasFactory, Notifiable, HasRoles;

    /**
     * Default guard for permissions
     */
    protected string $guard_name = 'staff';

    protected $fillable = [
        'staff_number',
        'full_name',
        'email',
        'phone',
        'department',
        'position',
        'staff_type',
        'employment_status',
        'termination_date',
        'notes',
        'password',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'termination_date' => 'date',
        'employment_status' => 'string',
        'staff_type' => 'string',
        'password' => 'hashed',
    ];

    /**
     * Generate next available staff number based on staff type
     * 
     * @param string $staffType 'senior' or 'junior'
     * @return string
     */
    public static function generateStaffNumber(string $staffType): string
    {
        $prefix = $staffType === 'senior' ? 'PN' : 'JN';
        
        // Get the latest staff number for this type
        $latestStaff = self::where('staff_number', 'LIKE', $prefix . '%')
            ->orderBy('staff_number', 'desc')
            ->first();

        if (!$latestStaff) {
            // First staff of this type
            return $prefix . '001';
        }

        // Extract number part and increment
        $number = (int) substr($latestStaff->staff_number, 2);
        $nextNumber = $number + 1;

        return $prefix . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
    }

    /**
     * Validate staff number format
     * 
     * @param string $staffNumber
     * @return bool
     */
    public static function isValidStaffNumber(string $staffNumber): bool
    {
        return preg_match('/^(PN|JN)\d{3}$/', $staffNumber);
    }

    /**
     * Get staff type from staff number
     * 
     * @param string $staffNumber
     * @return string|null
     */
    public static function getStaffTypeFromNumber(string $staffNumber): ?string
    {
        if (str_starts_with($staffNumber, 'PN')) {
            return 'senior';
        } elseif (str_starts_with($staffNumber, 'JN')) {
            return 'junior';
        }
        
        return null;
    }

    /**
     * Scope for active staff
     */
    public function scopeActive($query)
    {
        return $query->where('employment_status', 'active');
    }

    /**
     * Scope for senior staff
     */
    public function scopeSenior($query)
    {
        return $query->where('staff_type', 'senior');
    }

    /**
     * Scope for junior staff
     */
    public function scopeJunior($query)
    {
        return $query->where('staff_type', 'junior');
    }

    /**
     * Check if staff is currently employed
     */
    public function getIsEmployedAttribute(): bool
    {
        return $this->employment_status === 'active' && !$this->termination_date;
    }

    /**
     * Get the exams this staff member is assigned to as an examiner.
     */
    public function assignedExams(): \Illuminate\Database\Eloquent\Relations\BelongsToMany
    {
        return $this->belongsToMany(Exam::class, 'exam_examiners', 'staff_id', 'exam_id')
            ->withPivot(['role', 'can_grade', 'can_edit_questions'])
            ->withTimestamps();
    }

    /**
     * Get exams where this staff can grade essays.
     */
    public function gradableExams(): \Illuminate\Database\Eloquent\Relations\BelongsToMany
    {
        return $this->assignedExams()->wherePivot('can_grade', true);
    }
}
