<?php

namespace App\Livewire\Staff;

use App\Models\ExamAnswer;
use App\Models\ExamAttempt;
use App\Settings\TerminologySettings;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('components.layouts.app')]
class EssayGrading extends Component
{
    use WithPagination;

    public ?int $selectedAnswerId = null;
    public ?ExamAnswer $currentAnswer = null;
    public float $marksAwarded = 0;
    public string $feedback = '';
    public string $filterStatus = 'pending';

    public function mount(): void
    {
        $staff = Auth::guard('staff')->user();
        
        // Check if staff has permission to grade
        if (!$staff->hasAnyRole(['Medical Director', 'Examiner']) && !$staff->hasPermissionTo('grade exams')) {
            abort(403, 'You do not have permission to grade essays.');
        }
    }

    public function getStaff()
    {
        return Auth::guard('staff')->user();
    }

    public function getPendingCountProperty(): int
    {
        return $this->getAccessibleAnswersQuery()
            ->where('is_graded', false)
            ->count();
    }

    public function getGradedCountProperty(): int
    {
        return $this->getAccessibleAnswersQuery()
            ->where('is_graded', true)
            ->count();
    }

    protected function getAccessibleAnswersQuery()
    {
        $staff = $this->getStaff();
        
        $query = ExamAnswer::query()
            ->where('requires_grading', true);

        // If staff is Medical Director or has grade exams permission, they can grade all
        $canGradeAll = $staff->hasAnyRole(['Medical Director']) || $staff->hasPermissionTo('grade exams');

        if (!$canGradeAll) {
            // Examiners can only grade exams they are assigned to
            $examIds = $staff->assignedExams()
                ->wherePivot('can_grade', true)
                ->pluck('exams.id')
                ->toArray();

            if (!empty($examIds)) {
                $query->whereHas('attempt', function ($q) use ($examIds) {
                    $q->whereIn('exam_id', $examIds);
                });
            } else {
                $query->whereRaw('1 = 0'); // No access if no assigned exams
            }
        }

        return $query;
    }

    public function selectAnswer(int $answerId): void
    {
        $this->selectedAnswerId = $answerId;
        $this->currentAnswer = ExamAnswer::with(['question', 'attempt.examSession.exam', 'attempt.candidate', 'grader'])
            ->find($answerId);

        if ($this->currentAnswer) {
            $this->marksAwarded = $this->currentAnswer->marks_obtained ?? 0;
            $this->feedback = $this->currentAnswer->grading_feedback ?? '';
        }
    }

    public function submitGrade(): void
    {
        if (!$this->currentAnswer) {
            session()->flash('error', 'No answer selected for grading.');
            return;
        }

        $this->validate([
            'marksAwarded' => ['required', 'numeric', 'min:0', 'max:' . ($this->currentAnswer->marks_possible ?? 100)],
            'feedback' => ['nullable', 'string', 'max:2000'],
        ]);

        $staff = $this->getStaff();

        // Update the answer with grade
        $this->currentAnswer->update([
            'marks_obtained' => $this->marksAwarded,
            'grading_feedback' => $this->feedback,
            'is_graded' => true,
            'graded_by' => $staff->id,
            'graded_at' => now(),
            'is_correct' => $this->marksAwarded > 0,
        ]);

        // Update the exam attempt stats
        $this->currentAnswer->attempt->updateStats();

        session()->flash('success', 'Grade submitted successfully!');

        // Reset form
        $this->reset(['selectedAnswerId', 'currentAnswer', 'marksAwarded', 'feedback']);
    }

    public function skipAnswer(): void
    {
        $this->reset(['selectedAnswerId', 'currentAnswer', 'marksAwarded', 'feedback']);
    }

    public function setFilter(string $status): void
    {
        $this->filterStatus = $status;
        $this->resetPage();
    }

    public function render()
    {
        $terminology = app(TerminologySettings::class);
        
        $query = $this->getAccessibleAnswersQuery()
            ->with(['question', 'attempt.examSession.exam', 'attempt.candidate']);

        if ($this->filterStatus === 'pending') {
            $query->where('is_graded', false);
        } elseif ($this->filterStatus === 'graded') {
            $query->where('is_graded', true);
        }

        $answers = $query->orderBy('created_at', 'asc')->paginate(20);

        return view('livewire.staff.essay-grading', [
            'staff' => $this->getStaff(),
            'terminology' => $terminology,
            'answers' => $answers,
            'pendingCount' => $this->pendingCount,
            'gradedCount' => $this->gradedCount,
        ]);
    }
}
