<?php

namespace App\Livewire\Setup;

use App\Models\User;
use App\Settings\BrandingSettings;
use App\Settings\ExamSettings;
use App\Settings\FeatureSettings;
use App\Settings\OrganizationSettings;
use App\Settings\TerminologySettings;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;

#[Layout('components.layouts.setup')]
#[Title('Setup Wizard - CBT System')]
class SetupWizard extends Component
{
    use WithFileUploads;

    // Current step in the wizard
    public int $currentStep = 1;
    public int $totalSteps = 6;

    // Step 1: Organization Info
    #[Validate('required|string|max:255')]
    public string $organizationName = '';

    #[Validate('nullable|string|max:255')]
    public ?string $tagline = '';

    #[Validate('nullable|email|max:255')]
    public ?string $contactEmail = '';

    #[Validate('nullable|string|max:50')]
    public ?string $contactPhone = '';

    #[Validate('required|string|max:100')]
    public string $timezone = 'Africa/Lagos';

    // Step 2: Branding
    public $logo;

    #[Validate('required|string|regex:/^#[0-9A-Fa-f]{6}$/')]
    public string $primaryColor = '#2563eb';

    #[Validate('required|string|regex:/^#[0-9A-Fa-f]{6}$/')]
    public string $secondaryColor = '#4b5563';

    #[Validate('required|string|regex:/^#[0-9A-Fa-f]{6}$/')]
    public string $accentColor = '#059669';

    // Step 3: User Type Preset Selection
    #[Validate('required|string')]
    public string $selectedPreset = 'educational';

    // Step 4: Terminology (can be customized after preset selection)
    #[Validate('required|string|max:50')]
    public string $primaryUserSingular = 'Student';

    #[Validate('required|string|max:50')]
    public string $primaryUserPlural = 'Students';

    #[Validate('required|string|max:50')]
    public string $primaryUserIdLabel = 'Matriculation Number';

    #[Validate('required|string|max:50')]
    public string $secondaryUserSingular = 'Staff';

    #[Validate('required|string|max:50')]
    public string $secondaryUserPlural = 'Staff';

    #[Validate('required|string|max:50')]
    public string $secondaryUserIdLabel = 'Staff Number';

    #[Validate('required|string|max:50')]
    public string $periodLabel = 'Academic Session';

    #[Validate('required|string|max:50')]
    public string $divisionLabel = 'Department';

    // Step 5: Features
    public bool $featureSecondaryUserExams = true;
    public bool $featurePracticeMode = false;
    public bool $featureCertificateGeneration = false;
    public bool $featureResultPrinting = true;
    public bool $featureBulkImport = true;
    public bool $featureDarkMode = true;

    // Step 6: Admin Account
    #[Validate('required|string|max:255')]
    public string $adminName = '';

    #[Validate('required|email|max:255|unique:users,email')]
    public string $adminEmail = '';

    #[Validate('required|string|min:8')]
    public string $adminPassword = '';

    #[Validate('required|string|same:adminPassword')]
    public string $adminPasswordConfirmation = '';

    public function mount(): void
    {
        // Load defaults from config
        $defaults = config('cbt.defaults');

        $this->organizationName = $defaults['organization']['name'];
        $this->tagline = $defaults['organization']['tagline'];
        $this->timezone = $defaults['organization']['timezone'];

        $this->primaryColor = $defaults['branding']['primary_color'];
        $this->secondaryColor = $defaults['branding']['secondary_color'];
        $this->accentColor = $defaults['branding']['accent_color'];

        // Apply default preset
        $this->applyPreset('educational');
    }

    public function applyPreset(string $preset): void
    {
        $presetData = config("cbt.user_type_presets.{$preset}");

        if (! $presetData) {
            return;
        }

        $this->selectedPreset = $preset;
        $this->primaryUserSingular = $presetData['primary']['singular'];
        $this->primaryUserPlural = $presetData['primary']['plural'];
        $this->primaryUserIdLabel = $presetData['primary']['id_label'];
        $this->secondaryUserSingular = $presetData['secondary']['singular'];
        $this->secondaryUserPlural = $presetData['secondary']['plural'];
        $this->secondaryUserIdLabel = $presetData['secondary']['id_label'];
        $this->periodLabel = $presetData['period_label'];
        $this->divisionLabel = $presetData['division_label'];
    }

    public function updatedSelectedPreset(string $value): void
    {
        $this->applyPreset($value);
    }

    public function nextStep(): void
    {
        $this->validateCurrentStep();

        if ($this->currentStep < $this->totalSteps) {
            $this->currentStep++;
        }
    }

    public function previousStep(): void
    {
        if ($this->currentStep > 1) {
            $this->currentStep--;
        }
    }

    public function goToStep(int $step): void
    {
        // Only allow going to previous steps or current step
        if ($step <= $this->currentStep && $step >= 1) {
            $this->currentStep = $step;
        }
    }

    protected function validateCurrentStep(): void
    {
        match ($this->currentStep) {
            1 => $this->validate([
                'organizationName' => 'required|string|max:255',
                'tagline' => 'nullable|string|max:255',
                'contactEmail' => 'nullable|email|max:255',
                'contactPhone' => 'nullable|string|max:50',
                'timezone' => 'required|string|max:100',
            ]),
            2 => $this->validate([
                'logo' => 'nullable|image|max:2048',
                'primaryColor' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
                'secondaryColor' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
                'accentColor' => 'required|string|regex:/^#[0-9A-Fa-f]{6}$/',
            ]),
            3 => $this->validate([
                'selectedPreset' => 'required|string',
            ]),
            4 => $this->validate([
                'primaryUserSingular' => 'required|string|max:50',
                'primaryUserPlural' => 'required|string|max:50',
                'primaryUserIdLabel' => 'required|string|max:50',
                'secondaryUserSingular' => 'required|string|max:50',
                'secondaryUserPlural' => 'required|string|max:50',
                'secondaryUserIdLabel' => 'required|string|max:50',
                'periodLabel' => 'required|string|max:50',
                'divisionLabel' => 'required|string|max:50',
            ]),
            5 => true, // Features are optional, no validation needed
            6 => $this->validate([
                'adminName' => 'required|string|max:255',
                'adminEmail' => 'required|email|max:255|unique:users,email',
                'adminPassword' => ['required', 'string', Password::min(8)->mixedCase()->numbers()],
                'adminPasswordConfirmation' => 'required|string|same:adminPassword',
            ]),
            default => true,
        };
    }

    public function completeSetup(): void
    {
        $this->validateCurrentStep();

        // Save Organization Settings
        $orgSettings = app(OrganizationSettings::class);
        $orgSettings->name = $this->organizationName;
        $orgSettings->tagline = $this->tagline;
        $orgSettings->contact_email = $this->contactEmail;
        $orgSettings->contact_phone = $this->contactPhone;
        $orgSettings->timezone = $this->timezone;
        $orgSettings->save();

        // Save Branding Settings
        $brandingSettings = app(BrandingSettings::class);
        if ($this->logo) {
            $logoPath = $this->logo->store('branding', 'public');
            $brandingSettings->logo_path = $logoPath;
        }
        $brandingSettings->primary_color = $this->primaryColor;
        $brandingSettings->secondary_color = $this->secondaryColor;
        $brandingSettings->accent_color = $this->accentColor;
        $brandingSettings->save();

        // Save Terminology Settings
        $terminologySettings = app(TerminologySettings::class);
        $terminologySettings->primary_user_singular = $this->primaryUserSingular;
        $terminologySettings->primary_user_plural = $this->primaryUserPlural;
        $terminologySettings->primary_user_id_label = $this->primaryUserIdLabel;
        $terminologySettings->primary_user_id_field = 'matriculation_number';
        $terminologySettings->secondary_user_singular = $this->secondaryUserSingular;
        $terminologySettings->secondary_user_plural = $this->secondaryUserPlural;
        $terminologySettings->secondary_user_id_label = $this->secondaryUserIdLabel;
        $terminologySettings->secondary_user_id_field = 'staff_number';
        $terminologySettings->period_label = $this->periodLabel;
        $terminologySettings->period_format = 'YYYY/YYYY';
        $terminologySettings->division_label = $this->divisionLabel;
        $terminologySettings->save();

        // Save Feature Settings
        $featureSettings = app(FeatureSettings::class);
        $featureSettings->secondary_user_exams = $this->featureSecondaryUserExams;
        $featureSettings->practice_mode = $this->featurePracticeMode;
        $featureSettings->certificate_generation = $this->featureCertificateGeneration;
        $featureSettings->result_printing = $this->featureResultPrinting;
        $featureSettings->bulk_import = $this->featureBulkImport;
        $featureSettings->dark_mode = $this->featureDarkMode;
        $featureSettings->save();

        // Create Admin User
        $admin = User::create([
            'name' => $this->adminName,
            'email' => $this->adminEmail,
            'password' => Hash::make($this->adminPassword),
            'email_verified_at' => now(),
        ]);

        // Assign Super Admin role if Spatie Permission is configured
        if (method_exists($admin, 'assignRole')) {
            // Create role if it doesn't exist
            $roleClass = config('permission.models.role');
            $roleClass::firstOrCreate(['name' => 'Super Admin', 'guard_name' => 'web']);
            $admin->assignRole('Super Admin');
        }

        // Mark setup as complete
        $orgSettings->setup_completed = true;
        $orgSettings->save();

        // Redirect to login
        $this->redirect(route('login'), navigate: true);
    }

    public function getAvailablePresets(): array
    {
        return config('cbt.user_type_presets', []);
    }

    public function getTimezones(): array
    {
        return collect(timezone_identifiers_list())
            ->mapWithKeys(fn ($tz) => [$tz => $tz])
            ->all();
    }

    public function getFeatureDescriptions(): array
    {
        return FeatureSettings::getFeatureDescriptions();
    }

    public function render()
    {
        return view('livewire.setup.setup-wizard', [
            'presets' => $this->getAvailablePresets(),
            'timezones' => $this->getTimezones(),
            'featureDescriptions' => $this->getFeatureDescriptions(),
        ]);
    }
}
