<?php

namespace App\Livewire\Exam;

use App\Enums\ExamAttemptStatus;
use App\Enums\ExamSessionStatus;
use App\Models\ExamAnswer;
use App\Models\ExamAttempt;
use App\Models\ExamSession;
use App\Models\QuestionBank;
use App\Settings\TerminologySettings;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Component;

class TakeExam extends Component
{
    public ExamSession $session;
    public ?ExamAttempt $attempt = null;
    public ?QuestionBank $currentQuestion = null;
    public ?string $selectedAnswer = null;
    public ?string $essayAnswer = null;
    public array $navigationData = [];
    public int $currentIndex = 0;
    public int $timeRemaining = 0;
    public bool $showConfirmSubmit = false;
    public bool $isSubmitting = false;
    public string $accessCode = '';
    public bool $needsAccessCode = false;
    public bool $examStarted = false;

    public function mount(ExamSession $session)
    {
        $this->session = $session;
        
        $candidateType = $this->getCandidateType();
        $candidateId = $this->getCandidate()->id;

        // Check for existing in-progress attempt first
        $existingAttempt = $session->attempts()
            ->where('candidate_type', $candidateType)
            ->where('candidate_id', $candidateId)
            ->where('status', ExamAttemptStatus::IN_PROGRESS)
            ->first();

        if ($existingAttempt) {
            $this->attempt = $existingAttempt;
            $this->examStarted = true;
            $this->needsAccessCode = false;
            $this->loadQuestionAtIndex($existingAttempt->current_question_index);
            $this->updateNavigationData();
            $this->calculateTimeRemaining();
            return;
        }

        // Count completed attempts for this candidate
        $completedAttemptCount = $session->attempts()
            ->where('candidate_type', $candidateType)
            ->where('candidate_id', $candidateId)
            ->whereIn('status', [ExamAttemptStatus::COMPLETED, ExamAttemptStatus::TIMED_OUT])
            ->count();

        // If max attempts reached, redirect to the latest result
        if ($completedAttemptCount >= $session->max_attempts) {
            $latestAttempt = $session->attempts()
                ->where('candidate_type', $candidateType)
                ->where('candidate_id', $candidateId)
                ->whereIn('status', [ExamAttemptStatus::COMPLETED, ExamAttemptStatus::TIMED_OUT])
                ->latest()
                ->first();

            if ($latestAttempt) {
                return $this->redirect(
                    route($this->getExamResultRoute(), ['attempt' => $latestAttempt->id]),
                    navigate: true
                );
            }
        }

        // Check if access code is needed
        if ($session->access_code) {
            $this->needsAccessCode = true;
        }
    }

    public function verifyAccessCode()
    {
        if ($this->session->access_code && $this->accessCode !== $this->session->access_code) {
            $this->addError('accessCode', 'Invalid access code. Please try again.');
            return;
        }

        $this->needsAccessCode = false;
    }

    public function startExam()
    {
        $candidate = $this->getCandidate();
        $candidateType = $this->getCandidateType();

        // Verify candidate can take this exam
        $canTake = $this->session->canCandidateTake($candidateType, $candidate->id);
        
        if (!$canTake['can_take']) {
            session()->flash('error', implode(' ', $canTake['reasons']));
            return $this->redirect(route($this->getExamListRoute()), navigate: true);
        }

        // Resume existing attempt if any
        if ($canTake['in_progress_attempt']) {
            $this->attempt = $canTake['in_progress_attempt'];
        } else {
            // Create new attempt
            $questionOrder = $this->session->exam->getQuestionsForAttempt();
            
            if (empty($questionOrder)) {
                session()->flash('error', 'No questions available for this exam.');
                return $this->redirect(route($this->getExamListRoute()), navigate: true);
            }

            $this->attempt = ExamAttempt::create([
                'exam_session_id' => $this->session->id,
                'candidate_type' => $candidateType,
                'candidate_id' => $candidate->id,
                'question_order' => $questionOrder,
                'started_at' => now(),
                'total_questions' => count($questionOrder),
                'status' => ExamAttemptStatus::IN_PROGRESS,
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            // Pre-create answer records for all questions
            foreach ($questionOrder as $index => $questionId) {
                $question = QuestionBank::find($questionId);
                ExamAnswer::create([
                    'exam_attempt_id' => $this->attempt->id,
                    'question_bank_id' => $questionId,
                    'question_index' => $index + 1,
                    'correct_answer' => $question->correct_answer,
                    'marks_possible' => (int) $this->session->exam->marks_per_question,
                    'is_skipped' => true,
                ]);
            }
        }

        $this->examStarted = true;
        $this->loadQuestionAtIndex(0);
        $this->updateNavigationData();
        $this->calculateTimeRemaining();
    }

    public function getGuard(): string
    {
        return session('guard', 'student');
    }

    public function getCandidate(): object
    {
        return Auth::guard($this->getGuard())->user();
    }

    public function getCandidateType(): string
    {
        return match($this->getGuard()) {
            'student' => \App\Models\Student::class,
            'applicant' => \App\Models\Applicant::class,
            default => \App\Models\Staff::class,
        };
    }

    protected function getExamListRoute(): string
    {
        return match($this->getGuard()) {
            'student' => 'exam.list',
            'applicant' => 'applicant.exam.list',
            default => 'staff.exam.list',
        };
    }

    protected function getExamResultRoute(): string
    {
        return match($this->getGuard()) {
            'student' => 'exam.result',
            'applicant' => 'applicant.exam.result',
            default => 'staff.exam.result',
        };
    }

    public function loadQuestionAtIndex(int $index)
    {
        $this->currentIndex = $index;
        $this->currentQuestion = $this->attempt->getQuestionAtIndex($index);
        
        if ($this->currentQuestion) {
            // Load the existing answer if any
            $answer = $this->attempt->getAnswerForQuestion($this->currentQuestion->id);
            
            // Reset both answer types
            $this->selectedAnswer = null;
            $this->essayAnswer = null;
            
            // Load the appropriate answer based on question type
            if ($this->currentQuestion->isEssay()) {
                $this->essayAnswer = $answer?->essay_answer;
            } else {
                $this->selectedAnswer = $answer?->selected_answer;
            }
        }

        // Update current index in database
        $this->attempt->update(['current_question_index' => $index]);
    }

    public function selectAnswer(string $answer)
    {
        $this->selectedAnswer = $answer;
    }

    public function saveAndNext()
    {
        if (!$this->ensureAttemptIsValid()) return;

        $this->saveCurrentAnswer();
        
        if ($this->currentIndex < count($this->attempt->question_order) - 1) {
            $this->loadQuestionAtIndex($this->currentIndex + 1);
        }
        
        $this->updateNavigationData();
    }

    public function saveAndPrevious()
    {
        if (!$this->ensureAttemptIsValid()) return;

        $this->saveCurrentAnswer();
        
        if ($this->currentIndex > 0) {
            $this->loadQuestionAtIndex($this->currentIndex - 1);
        }
        
        $this->updateNavigationData();
    }

    public function goToQuestion(int $index)
    {
        if (!$this->ensureAttemptIsValid()) return;

        $this->saveCurrentAnswer();
        $this->loadQuestionAtIndex($index);
        $this->updateNavigationData();
    }

    public function saveCurrentAnswer()
    {
        if (!$this->currentQuestion) return;
        if (!$this->attempt || $this->attempt->status !== ExamAttemptStatus::IN_PROGRESS) return;

        // Pass essay answer for essay questions
        $this->attempt->submitAnswer(
            $this->currentQuestion->id,
            $this->selectedAnswer,
            0,
            $this->currentQuestion->isEssay() ? $this->essayAnswer : null
        );
    }

    public function flagQuestion()
    {
        if (!$this->currentQuestion) return;
        if (!$this->ensureAttemptIsValid()) return;
        
        $answer = $this->attempt->getAnswerForQuestion($this->currentQuestion->id);
        if ($answer) {
            $answer->update(['is_flagged' => !$answer->is_flagged]);
            $this->updateNavigationData();
        }
    }

    public function updateNavigationData()
    {
        $this->navigationData = $this->attempt->getNavigationData();
    }

    public function calculateTimeRemaining()
    {
        if ($this->attempt) {
            $this->timeRemaining = $this->attempt->time_remaining;
        }
    }

    public function handleTimeExpired()
    {
        if (!$this->ensureAttemptIsValid()) return;
        
        $this->submitExam('timed_out');
    }

    public function confirmSubmit()
    {
        if (!$this->ensureAttemptIsValid()) return;

        $this->saveCurrentAnswer();
        $this->showConfirmSubmit = true;
    }

    public function cancelSubmit()
    {
        $this->showConfirmSubmit = false;
    }

    /**
     * Check if the current attempt is still valid (in progress).
     * Redirects to results if attempt is already completed.
     */
    protected function ensureAttemptIsValid(): bool
    {
        if (!$this->attempt) {
            return true; // No attempt yet, allow starting
        }

        // Refresh the attempt from database to get latest status
        $this->attempt->refresh();

        if ($this->attempt->status !== ExamAttemptStatus::IN_PROGRESS) {
            // Attempt is no longer in progress - redirect to results
            $this->redirect(
                route($this->getExamResultRoute(), ['attempt' => $this->attempt->id]),
                navigate: true
            );
            return false;
        }

        return true;
    }

    public function submitExam(string $status = 'completed')
    {
        if (!$this->ensureAttemptIsValid()) {
            return;
        }

        $this->isSubmitting = true;
        $this->saveCurrentAnswer();
        
        $this->attempt->complete($status);

        // Close the modal first for smooth animation
        $this->showConfirmSubmit = false;

        // Get the results URL
        $resultsUrl = route($this->getExamResultRoute(), ['attempt' => $this->attempt->id]);

        // Dispatch browser event to redirect with history replacement
        // This prevents the back button from returning to the exam
        $this->dispatch('exam-submitted', url: $resultsUrl);
    }

    public function getProgressAttribute(): array
    {
        return $this->attempt?->getProgressSummary() ?? [];
    }

    public function render()
    {
        $guard = $this->getGuard();
        $user = $this->getCandidate();
        $terminology = app(TerminologySettings::class);
        
        $userRole = match($guard) {
            'student' => $terminology->primary_user_singular,
            'applicant' => $terminology->applicant_user_singular ?? 'Applicant',
            default => $terminology->secondary_user_singular,
        };
        
        return view('livewire.exam.take-exam', [
            'exam' => $this->session->exam,
            'progress' => $this->getProgressAttribute(),
        ])->layout('layouts.exam', [
            'title' => $this->session->exam->title ?? 'Exam',
            'userName' => $user->name ?? $user->full_name ?? 'Candidate',
            'userRole' => $userRole,
        ]);
    }
}
