<?php

namespace App\Livewire\Dashboard;

use App\Enums\ExamAttemptStatus;
use App\Enums\ExamSessionStatus;
use App\Models\ExamAttempt;
use App\Models\ExamSession;
use App\Settings\TerminologySettings;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Component;

#[Layout('components.layouts.app')]
class StudentDashboard extends Component
{
    public function getStudent()
    {
        return Auth::guard('student')->user();
    }

    public function getUpcomingExamsProperty(): array
    {
        $student = $this->getStudent();
        
        $sessions = ExamSession::with('exam')
            ->whereHas('exam', function ($q) {
                $q->where('is_active', true)
                  ->where(function ($q2) {
                      $q2->where('target_audience', 'student')
                         ->orWhere('target_audience', 'both');
                  });
            })
            ->where(function ($query) {
                $query->where('status', ExamSessionStatus::SCHEDULED)
                    ->orWhere('status', ExamSessionStatus::ACTIVE);
            })
            ->where('start_time', '>', now())
            ->accessibleBy(\App\Models\Student::class, $student->id)
            ->orderBy('start_time')
            ->limit(5)
            ->get();

        return $sessions->map(fn ($session) => [
            'id' => $session->id,
            'title' => $session->exam->title,
            'code' => $session->exam->code,
            'start_time' => $session->start_time,
            'duration' => $session->exam->duration_minutes,
            'total_questions' => $session->exam->total_questions,
        ])->toArray();
    }

    public function getAvailableExamsCountProperty(): int
    {
        $student = $this->getStudent();
        
        return ExamSession::whereHas('exam', function ($q) {
                $q->where('is_active', true)
                  ->where(function ($q2) {
                      $q2->where('target_audience', 'student')
                         ->orWhere('target_audience', 'both');
                  });
            })
            ->where('status', ExamSessionStatus::ACTIVE)
            ->where('start_time', '<=', now())
            ->where('end_time', '>=', now())
            ->accessibleBy(\App\Models\Student::class, $student->id)
            ->count();
    }

    public function getUpcomingExamsCountProperty(): int
    {
        $student = $this->getStudent();
        
        return ExamSession::whereHas('exam', function ($q) {
                $q->where('is_active', true)
                  ->where(function ($q2) {
                      $q2->where('target_audience', 'student')
                         ->orWhere('target_audience', 'both');
                  });
            })
            ->where(function ($query) {
                $query->where('status', ExamSessionStatus::SCHEDULED)
                    ->orWhere('status', ExamSessionStatus::ACTIVE);
            })
            ->where('start_time', '>', now())
            ->accessibleBy(\App\Models\Student::class, $student->id)
            ->count();
    }

    public function getCompletedExamsProperty(): array
    {
        $student = $this->getStudent();
        
        $attempts = ExamAttempt::with(['examSession.exam'])
            ->where('candidate_type', \App\Models\Student::class)
            ->where('candidate_id', $student->id)
            ->where('status', ExamAttemptStatus::COMPLETED)
            ->orderByDesc('completed_at')
            ->limit(5)
            ->get();

        return $attempts->map(fn ($attempt) => [
            'id' => $attempt->id,
            'exam_title' => $attempt->examSession->exam->title,
            'exam_code' => $attempt->examSession->exam->code,
            'completed_at' => $attempt->completed_at,
            'score_percentage' => $attempt->score_percentage,
            'passed' => $attempt->passed,
            'time_spent' => $this->formatDuration($attempt->time_spent_seconds),
        ])->toArray();
    }

    public function getCompletedExamsCountProperty(): int
    {
        $student = $this->getStudent();
        
        return ExamAttempt::where('candidate_type', \App\Models\Student::class)
            ->where('candidate_id', $student->id)
            ->where('status', ExamAttemptStatus::COMPLETED)
            ->count();
    }

    public function getAverageScoreProperty(): ?float
    {
        $student = $this->getStudent();
        
        $avg = ExamAttempt::where('candidate_type', \App\Models\Student::class)
            ->where('candidate_id', $student->id)
            ->where('status', ExamAttemptStatus::COMPLETED)
            ->avg('score_percentage');

        return $avg ? round($avg, 1) : null;
    }

    public function getPassRateProperty(): ?float
    {
        $student = $this->getStudent();
        
        $total = ExamAttempt::where('candidate_type', \App\Models\Student::class)
            ->where('candidate_id', $student->id)
            ->where('status', ExamAttemptStatus::COMPLETED)
            ->count();

        if ($total === 0) {
            return null;
        }

        $passed = ExamAttempt::where('candidate_type', \App\Models\Student::class)
            ->where('candidate_id', $student->id)
            ->where('status', ExamAttemptStatus::COMPLETED)
            ->where('passed', true)
            ->count();

        return round(($passed / $total) * 100, 1);
    }

    public function getInProgressAttemptProperty(): ?array
    {
        $student = $this->getStudent();
        
        $attempt = ExamAttempt::with(['examSession.exam'])
            ->where('candidate_type', \App\Models\Student::class)
            ->where('candidate_id', $student->id)
            ->where('status', ExamAttemptStatus::IN_PROGRESS)
            ->first();

        if (!$attempt) {
            return null;
        }

        return [
            'id' => $attempt->id,
            'session_id' => $attempt->exam_session_id,
            'exam_title' => $attempt->examSession->exam->title,
            'started_at' => $attempt->started_at,
            'time_remaining' => $this->calculateTimeRemaining($attempt),
        ];
    }

    protected function calculateTimeRemaining(ExamAttempt $attempt): int
    {
        $exam = $attempt->examSession->exam;
        $durationSeconds = $exam->duration_minutes * 60;
        $elapsedSeconds = now()->diffInSeconds($attempt->started_at);
        
        return max(0, $durationSeconds - $elapsedSeconds);
    }

    protected function formatDuration(int $seconds): string
    {
        $minutes = floor($seconds / 60);
        $remainingSeconds = $seconds % 60;
        
        if ($minutes >= 60) {
            $hours = floor($minutes / 60);
            $mins = $minutes % 60;
            return "{$hours}h {$mins}m";
        }
        
        return "{$minutes}m {$remainingSeconds}s";
    }

    public function render()
    {
        $terminology = app(TerminologySettings::class);
        $examSettings = app(\App\Settings\ExamSettings::class);
        
        return view('livewire.dashboard.student-dashboard', [
            'student' => $this->getStudent(),
            'terminology' => $terminology,
            'upcomingExams' => $this->upcomingExams,
            'completedExams' => $this->completedExams,
            'availableExamsCount' => $this->availableExamsCount,
            'upcomingExamsCount' => $this->upcomingExamsCount,
            'completedExamsCount' => $this->completedExamsCount,
            'averageScore' => $this->averageScore,
            'passRate' => $this->passRate,
            'inProgressAttempt' => $this->inProgressAttempt,
            'showResults' => $examSettings->show_results_to_students,
        ]);
    }
}
