<?php

namespace App\Imports;

use App\Models\QuestionBank;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithBatchInserts;

class QuestionsImport implements ToModel, WithHeadingRow, WithValidation, WithChunkReading, WithBatchInserts
{
    private array $metadata;
    private string $batchId;

    public function __construct(array $metadata, string $batchId)
    {
        $this->metadata = $metadata;
        $this->batchId = $batchId;
    }

    /**
     * Transform each row into a model
     */
    public function model(array $row): ?QuestionBank
    {
        // Skip empty rows
        if (empty($row['question'])) {
            return null;
        }

        // Build options array
        $options = [];
        $questionType = strtolower($row['type'] ?? 'single_select');
        
        if ($questionType === 'true_false') {
            $options = ['true' => 'True', 'false' => 'False'];
        } else {
            foreach (['option_a', 'option_b', 'option_c', 'option_d'] as $key) {
                $optionKey = str_replace('option_', '', $key);
                if (!empty($row[$key])) {
                    $options[$optionKey] = $row[$key];
                }
            }
        }

        return new QuestionBank([
            'question_text' => $row['question'],
            'question_type' => $questionType === 'true_false' ? 'true_false' : 'single_select',
            'options' => $options,
            'correct_answer' => $row['correct_answer'] ?? 'a',
            'explanation' => $row['explanation'] ?? null,
            'marks' => (int) ($row['marks'] ?? 1),
            'time_limit' => isset($row['time_limit']) ? (int) $row['time_limit'] : null,
            'category_id' => $this->metadata['category_id'],
            'target_audience' => $this->metadata['target_audience'],
            'session' => $this->metadata['session'],
            'department' => $this->metadata['department'],
            'import_batch_id' => $this->batchId,
            'imported_at' => now(),
            'is_active' => true,
            'created_by' => auth()->id(),
            'updated_by' => auth()->id(),
        ]);
    }

    /**
     * Validation rules for each row
     */
    public function rules(): array
    {
        return [
            'question' => 'required|string|max:5000',
            'type' => 'nullable|in:single_select,true_false',
            'correct_answer' => 'required|string',
            'marks' => 'nullable|integer|min:1|max:10',
            'time_limit' => 'nullable|integer|min:1|max:3600',
        ];
    }

    /**
     * Custom attribute names for validation errors
     */
    public function customValidationAttributes(): array
    {
        return [
            'question' => 'Question Text',
            'type' => 'Question Type',
            'correct_answer' => 'Correct Answer',
        ];
    }

    /**
     * Chunk size for processing large files
     */
    public function chunkSize(): int
    {
        return 100;
    }

    /**
     * Batch size for database inserts
     */
    public function batchSize(): int
    {
        return 50;
    }
}