<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Student;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class StudentAuthController extends Controller
{
    /**
     * Show the student login form
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Handle student login
     */
    public function login(Request $request)
    {
        $request->validate([
            'matriculation_number' => ['required', 'string'],
            'password' => ['required', 'string'],
        ]);

        Log::info('Student login attempt', [
            'matriculation_number' => $request->matriculation_number,
            'password_length' => strlen($request->password),
        ]);

        $student = Student::where('matriculation_number', $request->matriculation_number)->first();

        Log::info('Student lookup result', [
            'found' => $student ? true : false,
            'student_id' => $student?->id,
            'student_name' => $student?->full_name,
            'has_password' => $student ? !empty($student->password) : false,
            'password_hash_preview' => $student ? substr($student->password, 0, 20) . '...' : null,
        ]);

        if (!$student) {
            Log::warning('Student not found', ['matriculation_number' => $request->matriculation_number]);
            throw ValidationException::withMessages([
                'matriculation_number' => ['Invalid matriculation number or password.'],
            ]);
        }

        $passwordCheck = Hash::check($request->password, $student->password);
        Log::info('Password check result', [
            'password_valid' => $passwordCheck,
            'input_password' => $request->password,
            'stored_hash' => $student->password,
        ]);

        if (!$passwordCheck) {
            Log::warning('Password mismatch for student', ['student_id' => $student->id]);
            throw ValidationException::withMessages([
                'matriculation_number' => ['Invalid matriculation number or password.'],
            ]);
        }

        if (!$student->is_active) {
            Log::warning('Inactive student login attempt', ['student_id' => $student->id]);
            throw ValidationException::withMessages([
                'matriculation_number' => ['Your account has been deactivated. Please contact administration.'],
            ]);
        }

        Log::info('Student login successful', ['student_id' => $student->id]);

        // Log the student in
        Auth::guard('student')->login($student, $request->boolean('remember'));

        // Store the guard type in session
        $request->session()->put('guard', 'student');

        // Log the activity
        activity('student')
            ->causedBy($student)
            ->performedOn($student)
            ->withProperties([
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ])
            ->log('Student logged in');

        $request->session()->regenerate();

        return redirect()->intended(route('student.dashboard'));
    }

    /**
     * Handle student logout
     */
    public function logout(Request $request)
    {
        $student = Auth::guard('student')->user();

        if ($student) {
            // Log the activity
            activity('student')
                ->causedBy($student)
                ->performedOn($student)
                ->withProperties([
                    'ip' => $request->ip(),
                ])
                ->log('Student logged out');
        }

        Auth::guard('student')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('success', 'You have been logged out successfully.');
    }
}