<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Staff;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class StaffAuthController extends Controller
{
    /**
     * Show the staff login form
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Handle staff login
     */
    public function login(Request $request)
    {
        $request->validate([
            'staff_number' => ['required', 'string'],
            'password' => ['required', 'string'],
        ]);

        $staff = Staff::where('staff_number', $request->staff_number)->first();

        if (!$staff || !Hash::check($request->password, $staff->password ?? '')) {
            throw ValidationException::withMessages([
                'staff_number' => ['Invalid staff number or password.'],
            ]);
        }

        if ($staff->employment_status !== 'active') {
            throw ValidationException::withMessages([
                'staff_number' => ['Your account is not active. Please contact administration.'],
            ]);
        }

        // Log the staff in
        Auth::guard('staff')->login($staff, $request->boolean('remember'));

        // Store the guard type in session
        $request->session()->put('guard', 'staff');

        // Log the activity
        activity('staff')
            ->causedBy($staff)
            ->performedOn($staff)
            ->withProperties([
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ])
            ->log('Staff logged in');

        $request->session()->regenerate();

        // Redirect based on staff type or role
        if ($this->isAdmin($staff)) {
            return redirect()->intended('/admin');
        }

        return redirect()->intended(route('staff.dashboard'));
    }

    /**
     * Handle staff logout
     */
    public function logout(Request $request)
    {
        $staff = Auth::guard('staff')->user();

        if ($staff) {
            // Log the activity
            activity('staff')
                ->causedBy($staff)
                ->performedOn($staff)
                ->withProperties([
                    'ip' => $request->ip(),
                ])
                ->log('Staff logged out');
        }

        Auth::guard('staff')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('success', 'You have been logged out successfully.');
    }

    /**
     * Check if staff member has admin privileges
     */
    private function isAdmin(Staff $staff): bool
    {
        // Check if staff has the Examiner role (full access)
        return $staff->hasRole('Examiner');
    }
}