<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Applicant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class ApplicantAuthController extends Controller
{
    /**
     * Show the applicant login form
     */
    public function showLoginForm()
    {
        return view('auth.unified-login');
    }

    /**
     * Handle applicant login
     */
    public function login(Request $request)
    {
        $request->validate([
            'application_number' => ['required', 'string'],
            'password' => ['required', 'string'],
        ]);

        $applicant = Applicant::where('application_number', $request->application_number)->first();

        if (!$applicant || !Hash::check($request->password, $applicant->password ?? '')) {
            throw ValidationException::withMessages([
                'application_number' => ['Invalid application number or password.'],
            ]);
        }

        if (!$applicant->is_active) {
            throw ValidationException::withMessages([
                'application_number' => ['Your account is not active. Please contact administration.'],
            ]);
        }

        // Log the applicant in
        Auth::guard('applicant')->login($applicant, $request->boolean('remember'));

        // Store the guard type in session
        $request->session()->put('guard', 'applicant');

        // Log the activity
        activity('applicant')
            ->causedBy($applicant)
            ->performedOn($applicant)
            ->withProperties([
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ])
            ->log('Applicant logged in');

        $request->session()->regenerate();

        return redirect()->intended(route('applicant.dashboard'));
    }

    /**
     * Handle applicant logout
     */
    public function logout(Request $request)
    {
        $applicant = Auth::guard('applicant')->user();

        if ($applicant) {
            // Log the activity
            activity('applicant')
                ->causedBy($applicant)
                ->performedOn($applicant)
                ->withProperties([
                    'ip' => $request->ip(),
                ])
                ->log('Applicant logged out');
        }

        Auth::guard('applicant')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')->with('success', 'You have been logged out successfully.');
    }
}
