<?php

namespace App\Filament\Widgets;

use App\Models\ExamAttempt;
use App\Models\ExamSession;
use App\Enums\ExamAttemptStatus;
use App\Enums\ExamSessionStatus;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Widgets\TableWidget as BaseWidget;

class RecentExamAttempts extends BaseWidget
{
    protected static ?string $heading = 'Recent Exam Attempts';
    protected static ?int $sort = 3;
    protected int | string | array $columnSpan = 'full';

    public function table(Table $table): Table
    {
        return $table
            ->query(
                ExamAttempt::query()
                    ->with(['examSession.exam', 'candidate'])
                    ->latest('completed_at')
            )
            ->columns([
                Tables\Columns\TextColumn::make('candidate.full_name')
                    ->label('Candidate')
                    ->default(fn ($record) => $record->candidate?->full_name ?? $record->candidate?->name ?? 'Unknown')
                    ->searchable(),
                    
                Tables\Columns\TextColumn::make('examSession.exam.title')
                    ->label('Exam')
                    ->limit(30),
                    
                Tables\Columns\TextColumn::make('score_percentage')
                    ->label('Score')
                    ->suffix('%')
                    ->color(fn ($record) => $record->passed ? 'success' : 'danger'),
                    
                Tables\Columns\IconColumn::make('passed')
                    ->label('Result')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                    
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn ($state) => match($state) {
                        ExamAttemptStatus::COMPLETED => 'success',
                        ExamAttemptStatus::IN_PROGRESS => 'warning',
                        ExamAttemptStatus::TIMED_OUT => 'danger',
                        default => 'gray',
                    }),
                    
                Tables\Columns\TextColumn::make('completed_at')
                    ->label('Completed')
                    ->dateTime('M d, H:i')
                    ->sortable(),
            ])
            ->defaultSort('completed_at', 'desc')
            ->paginated([5, 10, 25]);
    }
}
