<?php

namespace App\Filament\Resources\Students\Tables;

use App\Settings\TerminologySettings;
use Filament\Actions\Action;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Support\Facades\Auth;

class StudentsTable
{
    public static function configure(Table $table): Table
    {
        $terminology = app(TerminologySettings::class);
        
        return $table
            ->columns([
                TextColumn::make('full_name')
                    ->label($terminology->primary_user_singular . ' Name')
                    ->searchable()
                    ->sortable()
                    ->weight('medium')
                    ->copyable(),

                TextColumn::make('matriculation_number')
                    ->label($terminology->primary_user_id_label)
                    ->searchable()
                    ->sortable()
                    ->copyable()
                    ->badge()
                    ->color('sea-green'),

                TextColumn::make('session')
                    ->label($terminology->period_label)
                    ->sortable()
                    ->badge()
                    ->color('citron'),

                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->toggleable()
                    ->placeholder('No email provided'),

                TextColumn::make('is_active')
                    ->label('Status')
                    ->badge()
                    ->formatStateUsing(fn ($state): string => $state ? 'Active' : 'Inactive')
                    ->color(fn ($state): string => $state ? 'success' : 'danger'),

                TextColumn::make('created_at')
                    ->label('Registered')
                    ->dateTime('M j, Y g:i A')
                    ->sortable()
                    ->toggleable(),
            ])
            ->filters([
                SelectFilter::make('session')
                    ->label($terminology->period_label)
                    ->options(function () {
                        $currentYear = (int) date('Y');
                        $sessions = [];
                        
                        // Generate sessions from 2 years ago to 5 years ahead
                        for ($year = $currentYear - 2; $year <= $currentYear + 5; $year++) {
                            $nextYear = $year + 1;
                            $shortYear = substr((string) $year, -2);
                            $shortNextYear = substr((string) $nextYear, -2);
                            $sessions["{$year}/{$shortNextYear}"] = "{$year}/{$nextYear}";
                        }
                        
                        return $sessions;
                    }),

                SelectFilter::make('is_active')
                    ->label('Status')
                    ->options([
                        '1' => 'Active',
                        '0' => 'Inactive',
                    ]),
            ])
            ->recordActions([
                ViewAction::make(),
                EditAction::make(),
                Action::make('toggle_status')
                    ->label(fn ($record): string => $record->is_active ? 'Deactivate' : 'Activate')
                    ->icon(fn ($record): string => $record->is_active ? 'heroicon-m-x-circle' : 'heroicon-m-check-circle')
                    ->color(fn ($record): string => $record->is_active ? 'danger' : 'success')
                    ->requiresConfirmation()
                    ->action(function ($record) use ($terminology) {
                        $record->update(['is_active' => !$record->is_active]);
                        
                        // Log the activity
                        activity('student')
                            ->causedBy(Auth::user())
                            ->performedOn($record)
                            ->withProperties([
                                'action' => $record->is_active ? 'activated' : 'deactivated',
                                'ip' => request()->ip(),
                            ])
                            ->log($record->is_active ? $terminology->primary_user_singular . ' activated' : $terminology->primary_user_singular . ' deactivated');
                    }),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    \Filament\Actions\BulkAction::make('activate')
                        ->label('Activate Selected')
                        ->icon('heroicon-m-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->action(function ($records) use ($terminology) {
                            $count = $records->count();
                            $records->each(fn ($record) => $record->update(['is_active' => true]));
                            
                            \Filament\Notifications\Notification::make()
                                ->title("{$count} " . $terminology->primary_user_plural . ' activated')
                                ->success()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion(),
                    
                    \Filament\Actions\BulkAction::make('deactivate')
                        ->label('Deactivate Selected')
                        ->icon('heroicon-m-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->action(function ($records) use ($terminology) {
                            $count = $records->count();
                            $records->each(fn ($record) => $record->update(['is_active' => false]));
                            
                            \Filament\Notifications\Notification::make()
                                ->title("{$count} " . $terminology->primary_user_plural . ' deactivated')
                                ->warning()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion(),
                    
                    DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->striped();
    }
}
