<?php

namespace App\Filament\Resources\Students\Schemas;

use App\Settings\TerminologySettings;
use Filament\Schemas\Schema;
use Filament\Schemas\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;

class StudentForm
{
    public static function configure(Schema $schema): Schema
    {
        $terminology = app(TerminologySettings::class);
        
        return $schema
            ->components([
                Section::make($terminology->primary_user_singular . ' Information')
                    ->description('Enter the ' . strtolower($terminology->primary_user_singular) . ' details for CBT system access')
                    ->schema([
                        TextInput::make('full_name')
                            ->label('Full Name')
                            ->required()
                            ->maxLength(255)
                            ->placeholder('e.g., John Doe Smith')
                            ->columnSpanFull(),

                        TextInput::make('matriculation_number')
                            ->label($terminology->primary_user_id_label)
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->maxLength(50)
                            ->placeholder('Enter ' . strtolower($terminology->primary_user_id_label))
                            ->helperText('Must be unique across the system')
                            ->suffixIcon('heroicon-m-identification'),

                        TextInput::make('email')
                            ->label('Email Address')
                            ->email()
                            ->unique(ignoreRecord: true)
                            ->maxLength(255)
                            ->placeholder('e.g., user@example.com')
                            ->helperText('Optional - Used for system notifications if provided')
                            ->suffixIcon('heroicon-m-envelope'),

                        Select::make('session')
                            ->label($terminology->period_label)
                            ->required()
                            ->options(function () {
                                $currentYear = (int) date('Y');
                                $sessions = [];
                                
                                // Generate sessions from 2 years ago to 5 years ahead
                                for ($year = $currentYear - 2; $year <= $currentYear + 5; $year++) {
                                    $nextYear = $year + 1;
                                    $shortYear = substr((string) $year, -2);
                                    $shortNextYear = substr((string) $nextYear, -2);
                                    $sessions["{$year}/{$shortNextYear}"] = "{$year}/{$nextYear}";
                                }
                                
                                return $sessions;
                            })
                            ->default(function () {
                                $currentYear = (int) date('Y');
                                $currentMonth = (int) date('n');
                                
                                // If we're in Jan-Aug, use previous academic year, else current
                                $academicYear = $currentMonth <= 8 ? $currentYear - 1 : $currentYear;
                                $nextYear = substr((string) ($academicYear + 1), -2);
                                
                                return "{$academicYear}/{$nextYear}";
                            })
                            ->searchable()
                            ->suffixIcon('heroicon-m-calendar-days'),

                        TextInput::make('password')
                            ->label('Password')
                            ->password()
                            ->required(fn (string $context): bool => $context === 'create')
                            ->minLength(6)
                            ->dehydrateStateUsing(fn (?string $state): ?string => 
                                filled($state) ? bcrypt($state) : null
                            )
                            ->dehydrated(fn (?string $state): bool => filled($state))
                            ->placeholder('Minimum 6 characters')
                            ->helperText($terminology->primary_user_singular . ' will use this to login to exams')
                            ->suffixIcon('heroicon-m-key')
                            ->columnSpanFull(),
                    ])
                    ->columns(3)
                    ->columnSpanFull()
                    ->collapsible()
                    ->persistCollapsed(false),
            ])
            ->columns(1); // Make the entire schema single column for full width
    }
}
