<?php

namespace App\Filament\Resources\QuestionBanks\Tables;

use App\Models\QuestionBank;
use App\Settings\TerminologySettings;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ViewAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;

class QuestionBanksTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(fn ($query) => $query
                ->with(['creator:id,name', 'updater:id,name', 'category:id,name']) // Eager load creators and category
                ->select([
                    'id', 'question_text', 'question_type', 'target_audience', 
                    'session', 'category_id', 'subject', 'time_limit', 
                    'is_active', 'created_at', 'updated_at', 'deleted_at',
                    'created_by', 'updated_by'
                ]) // Select only needed columns
            )
            ->columns([
                TextColumn::make('question_text')
                    ->label('Question')
                    ->limit(100)
                    ->searchable()
                    ->sortable(),

                TextColumn::make('question_type')
                    ->label('Type')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'single_select' => 'info',
                        'true_false' => 'warning',
                        default => 'gray',
                    })
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'single_select' => 'MCQ',
                        'true_false' => 'T/F',
                        default => $state,
                    }),

                TextColumn::make('target_audience')
                    ->label('Audience')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'student' => 'success',
                        'staff' => 'primary',
                        default => 'gray',
                    })
                    ->formatStateUsing(function (string $state): string {
                        $terminology = app(TerminologySettings::class);
                        return match ($state) {
                            'student' => $terminology->primary_user_singular,
                            'staff' => $terminology->secondary_user_singular,
                            default => ucfirst($state),
                        };
                    }),

                TextColumn::make('category.name')
                    ->label('Subject')
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('session')
                    ->label(fn () => app(TerminologySettings::class)->period_label)
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->alignCenter(),

                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('category_id')
                    ->label('Subject')
                    ->relationship('category', 'name')
                    ->searchable()
                    ->preload(),

                SelectFilter::make('question_type')
                    ->label('Question Type')
                    ->options(QuestionBank::getQuestionTypeOptions()),

                SelectFilter::make('target_audience')
                    ->label('Target Audience')
                    ->options(QuestionBank::getTargetAudienceOptions()),

                TernaryFilter::make('is_active')
                    ->label('Status')
                    ->placeholder('All Questions')
                    ->trueLabel('Active Only')
                    ->falseLabel('Inactive Only'),

                TrashedFilter::make(),
            ])
            ->actions([
                ViewAction::make(),
                EditAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->paginationPageOptions([25, 50, 100, 200]) // Pagination options for large datasets
            ->defaultPaginationPageOption(50) // Default to 50 items per page
            ->persistFiltersInSession() // Remember filters for better UX
            ->striped() // Alternate row colors for better readability
            ->poll('30s'); // Auto-refresh every 30 seconds for real-time updates
    }
}