<?php

namespace App\Filament\Resources\QuestionBanks\Pages;

use App\Filament\Resources\QuestionBanks\QuestionBankResource;
use App\Models\QuestionBank;
use App\Models\QuestionCategory;
use App\Services\QuestionImportService;
use App\Settings\TerminologySettings;
use Filament\Actions\CreateAction;
use Filament\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ListRecords;
use App\Services\TemplateGeneratorService;
use Illuminate\Support\Facades\Storage;

class ListQuestionBanks extends ListRecords
{
    protected static string $resource = QuestionBankResource::class;

    protected function getHeaderActions(): array
    {
        $terminology = app(TerminologySettings::class);
        
        return [
            Action::make('download_excel_template')
                ->label('Download Excel Template')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('gray')
                ->action(function () {
                    $templateService = app(TemplateGeneratorService::class);
                    $filePath = $templateService->generateExcelTemplate();
                    
                    return response()->download(
                        $filePath,
                        'question_import_template.xlsx',
                        ['Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet']
                    )->deleteFileAfterSend();
                }),
            
            Action::make('download_word_template')
                ->label('Download Word Template')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('gray')
                ->action(function () {
                    $templateService = app(TemplateGeneratorService::class);
                    $filePath = $templateService->generateWordTemplate();
                    
                    return response()->download(
                        $filePath,
                        'question_import_template.docx',
                        ['Content-Type' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document']
                    )->deleteFileAfterSend();
                }),
            
            Action::make('bulk_import')
                ->label('Bulk Import')
                ->icon('heroicon-o-arrow-up-tray')
                ->color('success')
                ->form([
                    FileUpload::make('file')
                        ->label('Question File')
                        ->acceptedFileTypes([
                            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', // .xlsx
                            'application/vnd.ms-excel', // .xls
                            'text/csv', // .csv
                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document', // .docx
                            'application/msword', // .doc
                        ])
                        ->maxSize(10240) // 10MB max
                        ->required()
                        ->helperText('Supported formats: Excel (.xlsx, .xls), CSV (.csv), Word (.docx, .doc). Maximum file size: 10MB'),
                    
                    Select::make('target_audience')
                        ->label('Target Audience')
                        ->options(QuestionBank::getTargetAudienceOptions())
                        ->required()
                        ->default('student'),
                    
                    TextInput::make('session')
                        ->label($terminology->period_label)
                        ->placeholder('e.g., 2024/2025')
                        ->helperText('Required for ' . strtolower($terminology->primary_user_singular) . ' questions'),
                    
                    Select::make('category_id')
                        ->label('Subject')
                        ->options(
                            QuestionCategory::where('is_active', true)
                                ->orderBy('name')
                                ->pluck('name', 'id')
                        )
                        ->searchable()
                        ->preload()
                        ->required()
                        ->helperText('All imported questions will be assigned to this subject'),
                    
                    TextInput::make('department')
                        ->label($terminology->division_label)
                        ->placeholder('Enter ' . strtolower($terminology->division_label)),
                    
                    Textarea::make('notes')
                        ->label('Import Notes')
                        ->placeholder('Any notes about this import batch')
                        ->rows(3),
                ])
                ->action(function (array $data) {
                    try {
                        $importService = app(QuestionImportService::class);
                        
                        // Debug: Log the file data to understand the structure
                        \Log::info('Import file data', ['file' => $data['file']]);
                        
                        // Handle the file path - Filament may return string or array
                        $fileName = is_array($data['file']) ? $data['file'][0] : $data['file'];
                        
                        // Try multiple possible locations
                        $possiblePaths = [
                            Storage::disk('local')->path($fileName),
                            Storage::disk('local')->path('private/' . $fileName),
                            Storage::disk('local')->path('livewire-tmp/' . $fileName),
                            storage_path('app/' . $fileName),
                            storage_path('app/private/' . $fileName),
                        ];
                        
                        $filePath = null;
                        foreach ($possiblePaths as $path) {
                            if (file_exists($path)) {
                                $filePath = $path;
                                break;
                            }
                        }
                        
                        if (!$filePath) {
                            \Log::error('File not found in any location', ['tried' => $possiblePaths]);
                            throw new \Exception('Uploaded file not found. Please try uploading again.');
                        }
                        
                        // Get file info
                        $originalName = basename($fileName);
                        $mimeType = mime_content_type($filePath);
                        
                        $uploadedFile = new \Illuminate\Http\UploadedFile(
                            $filePath,
                            $originalName,
                            $mimeType,
                            null,
                            true
                        );
                        
                        $metadata = [
                            'target_audience' => $data['target_audience'],
                            'session' => $data['session'] ?? null,
                            'category_id' => $data['category_id'] ?? null,
                            'department' => $data['department'] ?? null,
                            'notes' => $data['notes'] ?? null,
                        ];
                        
                        $result = $importService->processImport($uploadedFile, $metadata);
                        
                        // Clean up uploaded file
                        @unlink($filePath);
                        
                        if ($result['success']) {
                            Notification::make()
                                ->title('Import Successful')
                                ->body($result['message'])
                                ->success()
                                ->send();
                            
                            // Refresh the table without full page reload
                            $this->resetTable();
                        } else {
                            Notification::make()
                                ->title('Import Failed')
                                ->body($result['message'])
                                ->danger()
                                ->send();
                        }
                        
                    } catch (\Exception $e) {
                        Notification::make()
                            ->title('Import Error')
                            ->body('An error occurred during import: ' . $e->getMessage())
                            ->danger()
                            ->send();
                    }
                })
                ->modalHeading('Bulk Import Questions')
                ->modalDescription('Upload Excel, CSV, or Word documents containing questions. Imported questions will appear immediately in the list.')
                ->modalSubmitActionLabel('Start Import'),
            
            CreateAction::make(),
        ];
    }
}
