<?php

namespace App\Filament\Resources\ExamAttempts\Pages;

use App\Filament\Resources\ExamAttempts\ExamAttemptResource;
use Filament\Actions\Action;
use Filament\Resources\Pages\ListRecords;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use App\Settings\TerminologySettings;

class ListExamAttempts extends ListRecords
{
    protected static string $resource = ExamAttemptResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Action::make('export_all')
                ->label('Export All Results')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('success')
                ->action(function () {
                    // Get filtered records from the table
                    $records = $this->getFilteredTableQuery()->get();
                    
                    if ($records->isEmpty()) {
                        \Filament\Notifications\Notification::make()
                            ->title('No results to export')
                            ->warning()
                            ->send();
                        return;
                    }
                    
                    return $this->exportToExcel($records);
                }),
        ];
    }

    /**
     * Export exam attempts to Excel
     */
    protected function exportToExcel($records): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Exam Results');

        // Define headers
        $headers = [
            'A1' => 'Exam Code',
            'B1' => 'Exam Title',
            'C1' => 'Candidate Name',
            'D1' => 'Candidate ID',
            'E1' => 'Candidate Type',
            'F1' => 'Score (%)',
            'G1' => 'Correct Answers',
            'H1' => 'Total Questions',
            'I1' => 'Result',
            'J1' => 'Status',
            'K1' => 'Time Spent',
            'L1' => 'Started At',
            'M1' => 'Completed At',
        ];

        // Style headers
        foreach ($headers as $cell => $header) {
            $sheet->setCellValue($cell, $header);
            $sheet->getStyle($cell)->getFont()->setBold(true);
            $sheet->getStyle($cell)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setRGB('4F46E5');
            $sheet->getStyle($cell)->getFont()->getColor()->setRGB('FFFFFF');
            $sheet->getStyle($cell)->getAlignment()
                ->setHorizontal(Alignment::HORIZONTAL_CENTER);
        }

        // Set column widths
        $sheet->getColumnDimension('A')->setWidth(15);
        $sheet->getColumnDimension('B')->setWidth(30);
        $sheet->getColumnDimension('C')->setWidth(25);
        $sheet->getColumnDimension('D')->setWidth(15);
        $sheet->getColumnDimension('E')->setWidth(15);
        $sheet->getColumnDimension('F')->setWidth(12);
        $sheet->getColumnDimension('G')->setWidth(15);
        $sheet->getColumnDimension('H')->setWidth(15);
        $sheet->getColumnDimension('I')->setWidth(10);
        $sheet->getColumnDimension('J')->setWidth(12);
        $sheet->getColumnDimension('K')->setWidth(12);
        $sheet->getColumnDimension('L')->setWidth(18);
        $sheet->getColumnDimension('M')->setWidth(18);

        $terminology = app(TerminologySettings::class);

        // Add data
        $row = 2;
        foreach ($records as $attempt) {
            $minutes = floor($attempt->time_spent_seconds / 60);
            $seconds = $attempt->time_spent_seconds % 60;
            
            $sheet->setCellValue('A' . $row, $attempt->examSession?->exam?->code ?? '—');
            $sheet->setCellValue('B' . $row, $attempt->examSession?->exam?->title ?? '—');
            $sheet->setCellValue('C' . $row, $attempt->candidate_name);
            $sheet->setCellValue('D' . $row, $attempt->candidate_identifier);
            $sheet->setCellValue('E' . $row, str_contains($attempt->candidate_type, 'Student') 
                ? $terminology->secondary_user_singular 
                : $terminology->primary_user_singular);
            $sheet->setCellValue('F' . $row, number_format($attempt->score_percentage, 1));
            $sheet->setCellValue('G' . $row, $attempt->correct_answers);
            $sheet->setCellValue('H' . $row, $attempt->total_questions);
            $sheet->setCellValue('I' . $row, $attempt->passed ? 'PASS' : 'FAIL');
            $sheet->setCellValue('J' . $row, $attempt->status->getLabel());
            $sheet->setCellValue('K' . $row, "{$minutes}m {$seconds}s");
            $sheet->setCellValue('L' . $row, $attempt->started_at?->format('M d, Y H:i'));
            $sheet->setCellValue('M' . $row, $attempt->completed_at?->format('M d, Y H:i') ?? '—');

            // Color the result cell
            $resultColor = $attempt->passed ? '22C55E' : 'EF4444';
            $sheet->getStyle('I' . $row)->getFont()->getColor()->setRGB($resultColor);
            $sheet->getStyle('I' . $row)->getFont()->setBold(true);

            $row++;
        }

        // Auto-filter
        $sheet->setAutoFilter('A1:M' . ($row - 1));

        // Freeze header row
        $sheet->freezePane('A2');

        $filename = 'exam_results_' . now()->format('Y-m-d_His') . '.xlsx';

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }
}
