<?php

namespace App\Filament\Resources\Applicants\Pages;

use App\Filament\Resources\Applicants\ApplicantResource;
use App\Models\Applicant;
use Filament\Actions\Action;
use Filament\Actions\CreateAction;
use Filament\Forms\Components\FileUpload;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ListRecords;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;

class ListApplicants extends ListRecords
{
    protected static string $resource = ApplicantResource::class;

    protected function getHeaderActions(): array
    {
        return [
            CreateAction::make(),
            Action::make('download_template')
                ->label('Download Template')
                ->icon('heroicon-o-document-arrow-down')
                ->color('gray')
                ->url(route('applicants.template'))
                ->openUrlInNewTab(),
            Action::make('bulk_upload')
                ->label('Bulk Upload')
                ->icon('heroicon-o-arrow-up-tray')
                ->color('success')
                ->form([
                    FileUpload::make('file')
                        ->label('Upload CSV/Excel File')
                        ->acceptedFileTypes(['text/csv', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])
                        ->maxSize(2048)
                        ->required()
                        ->helperText('Upload a CSV or Excel file with columns: application_number, full_name, position_applied, department, email, phone, password. Use the "Download Template" button to get a sample file.')
                ])
                ->action(function (array $data): void {
                    $file = $data['file'];
                    $filePath = storage_path('app/private/' . $file);
                    
                    try {
                        $excelData = Excel::toArray([], $filePath)[0];
                        
                        if (empty($excelData)) {
                            Notification::make()
                                ->title('Error')
                                ->body('The uploaded file is empty.')
                                ->danger()
                                ->send();
                            return;
                        }

                        // Get headers from first row
                        $headers = array_shift($excelData);
                        $headers = array_map('trim', array_map('strtolower', $headers));

                        // Validate required headers
                        $requiredHeaders = ['application_number', 'full_name', 'position_applied', 'password'];
                        $missingHeaders = array_diff($requiredHeaders, $headers);
                        
                        if (!empty($missingHeaders)) {
                            Notification::make()
                                ->title('Error')
                                ->body('Missing required columns: ' . implode(', ', $missingHeaders))
                                ->danger()
                                ->send();
                            return;
                        }

                        $successCount = 0;
                        $errorCount = 0;
                        $errors = [];

                        foreach ($excelData as $index => $row) {
                            $rowNumber = $index + 2;
                            
                            // Skip empty rows
                            if (empty(array_filter($row))) {
                                continue;
                            }
                            
                            // Create associative array from row data
                            $applicantData = array_combine($headers, $row);
                            
                            // Validate row data
                            $validator = Validator::make($applicantData, [
                                'application_number' => [
                                    'required', 
                                    'string', 
                                    'max:50',
                                    Rule::unique('applicants', 'application_number')
                                ],
                                'full_name' => ['required', 'string', 'max:255'],
                                'position_applied' => ['required', 'string', 'max:255'],
                                'department' => ['nullable', 'string', 'max:255'],
                                'email' => [
                                    'nullable', 
                                    'email', 
                                    'max:255',
                                    Rule::unique('applicants', 'email')->whereNotNull('email')
                                ],
                                'phone' => ['nullable', 'string', 'max:20'],
                                'password' => ['required', 'string', 'min:6', 'max:255'],
                            ]);

                            if ($validator->fails()) {
                                $errorCount++;
                                $errors[] = "Row {$rowNumber}: " . implode(', ', $validator->errors()->all());
                                continue;
                            }

                            try {
                                Applicant::create([
                                    'application_number' => trim($applicantData['application_number']),
                                    'full_name' => trim($applicantData['full_name']),
                                    'position_applied' => trim($applicantData['position_applied']),
                                    'department' => isset($applicantData['department']) && !empty($applicantData['department']) ? trim($applicantData['department']) : null,
                                    'email' => isset($applicantData['email']) && !empty($applicantData['email']) ? trim($applicantData['email']) : null,
                                    'phone' => isset($applicantData['phone']) && !empty($applicantData['phone']) ? trim($applicantData['phone']) : null,
                                    'password' => trim($applicantData['password']),
                                    'is_active' => true,
                                ]);
                                $successCount++;
                            } catch (\Exception $e) {
                                $errorCount++;
                                $errors[] = "Row {$rowNumber}: Failed to create applicant - " . $e->getMessage();
                            }
                        }

                        // Show results
                        if ($successCount > 0) {
                            Notification::make()
                                ->title('Success')
                                ->body("Successfully imported {$successCount} applicants.")
                                ->success()
                                ->send();
                        }

                        if ($errorCount > 0) {
                            $errorMessage = "Failed to import {$errorCount} applicants.";
                            if (count($errors) <= 3) {
                                $errorMessage .= "\n\nErrors:\n" . implode("\n", array_slice($errors, 0, 3));
                            } else {
                                $errorMessage .= "\n\nShowing first 3 errors:\n" . implode("\n", array_slice($errors, 0, 3));
                                $errorMessage .= "\n... and " . (count($errors) - 3) . " more errors.";
                            }
                            
                            Notification::make()
                                ->title('Import Warnings')
                                ->body($errorMessage)
                                ->warning()
                                ->send();
                        }

                        if ($successCount === 0 && $errorCount === 0) {
                            Notification::make()
                                ->title('No Data')
                                ->body('No valid data rows found in the file.')
                                ->warning()
                                ->send();
                        }

                    } catch (\Exception $e) {
                        Notification::make()
                            ->title('Error')
                            ->body('Failed to process the file: ' . $e->getMessage())
                            ->danger()
                            ->send();
                    }

                    // Clean up the uploaded file
                    if (file_exists($filePath)) {
                        unlink($filePath);
                    }
                }),
        ];
    }
}
