<?php

namespace App\Filament\Pages;

use App\Settings\OrganizationSettings;
use Filament\Actions\Action;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use BackedEnum;
use UnitEnum;

class ManageOrganizationSettings extends Page
{
    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-building-office';

    protected string $view = 'filament.pages.manage-settings';

    protected static string|UnitEnum|null $navigationGroup = 'Settings';

    protected static ?int $navigationSort = 12;

    protected static ?string $title = 'Organization Settings';

    protected static ?string $navigationLabel = 'Organization';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(OrganizationSettings::class);
        
        $this->form->fill([
            'name' => $settings->name,
            'tagline' => $settings->tagline,
            'contact_email' => $settings->contact_email,
            'contact_phone' => $settings->contact_phone,
            'address' => $settings->address,
            'timezone' => $settings->timezone,
            'setup_completed' => $settings->setup_completed,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Organization Information')
                    ->description('Basic information about your organization.')
                    ->schema([
                        TextInput::make('name')
                            ->label('Organization Name')
                            ->required()
                            ->maxLength(255)
                            ->helperText('Full name of your organization'),

                        TextInput::make('tagline')
                            ->label('Tagline')
                            ->maxLength(255)
                            ->helperText('Short slogan or motto'),

                        TextInput::make('timezone')
                            ->label('Timezone')
                            ->required()
                            ->helperText('e.g., Africa/Lagos, UTC'),
                    ]),

                Section::make('Contact Information')
                    ->description('How candidates and staff can reach you.')
                    ->schema([
                        TextInput::make('contact_email')
                            ->label('Contact Email')
                            ->email()
                            ->maxLength(255),

                        TextInput::make('contact_phone')
                            ->label('Contact Phone')
                            ->tel()
                            ->maxLength(50),

                        Textarea::make('address')
                            ->label('Address')
                            ->rows(2)
                            ->maxLength(500),
                    ])
                    ->columns(2),

                Section::make('System Status')
                    ->description('System configuration status.')
                    ->schema([
                        Toggle::make('setup_completed')
                            ->label('Setup Completed')
                            ->helperText('Mark initial setup as complete'),
                    ]),
            ])
            ->statePath('data');
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label('Save Settings')
                ->submit('save'),
        ];
    }

    public function save(): void
    {
        $data = $this->form->getState();
        
        $settings = app(OrganizationSettings::class);
        $settings->name = $data['name'];
        $settings->tagline = $data['tagline'] ?? null;
        $settings->contact_email = $data['contact_email'] ?? null;
        $settings->contact_phone = $data['contact_phone'] ?? null;
        $settings->address = $data['address'] ?? null;
        $settings->timezone = $data['timezone'];
        $settings->setup_completed = (bool) $data['setup_completed'];
        $settings->save();

        Notification::make()
            ->title('Settings saved successfully')
            ->success()
            ->send();
    }
}
