<?php

namespace App\Filament\Pages;

use App\Settings\ExamSettings;
use Filament\Actions\Action;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use BackedEnum;
use UnitEnum;

class ManageExamSettings extends Page
{
    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-clipboard-document-check';

    protected string $view = 'filament.pages.manage-settings';

    protected static string|UnitEnum|null $navigationGroup = 'Settings';

    protected static ?int $navigationSort = 10;

    protected static ?string $title = 'Exam Settings';

    protected static ?string $navigationLabel = 'Exam Settings';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(ExamSettings::class);
        
        $this->form->fill([
            'default_duration_minutes' => $settings->default_duration_minutes,
            'default_passing_score' => $settings->default_passing_score,
            'max_attempts' => $settings->max_attempts,
            'shuffle_questions' => $settings->shuffle_questions,
            'shuffle_options' => $settings->shuffle_options,
            'allow_review' => $settings->allow_review,
            'show_score_immediately' => $settings->show_score_immediately,
            'show_results_to_students' => $settings->show_results_to_students,
            'show_correct_answers_to_students' => $settings->show_correct_answers_to_students,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Default Exam Configuration')
                    ->description('These are GLOBAL DEFAULTS applied when creating new exams. Each exam can override these values in its own "Exam Behavior" section.')
                    ->schema([
                        TextInput::make('default_duration_minutes')
                            ->label('Default Duration (minutes)')
                            ->numeric()
                            ->required()
                            ->minValue(5)
                            ->maxValue(480)
                            ->suffix('minutes')
                            ->helperText('Default exam duration when creating new exams'),

                        TextInput::make('default_passing_score')
                            ->label('Default Passing Score')
                            ->numeric()
                            ->required()
                            ->minValue(0)
                            ->maxValue(100)
                            ->suffix('%')
                            ->helperText('Minimum percentage required to pass'),

                        TextInput::make('max_attempts')
                            ->label('Maximum Attempts')
                            ->numeric()
                            ->required()
                            ->minValue(1)
                            ->maxValue(10)
                            ->helperText('Maximum number of attempts per exam session'),
                    ])
                    ->columns(3),

                Section::make('Question & Answer Options')
                    ->description('Default behavior for question presentation. Individual exams can override these settings.')
                    ->schema([
                        Toggle::make('shuffle_questions')
                            ->label('Shuffle Questions')
                            ->helperText('Randomize question order for each attempt'),

                        Toggle::make('shuffle_options')
                            ->label('Shuffle Answer Options')
                            ->helperText('Randomize answer option order for each question'),

                        Toggle::make('allow_review')
                            ->label('Allow Question Review')
                            ->helperText('Allow candidates to navigate back to previous questions'),
                    ])
                    ->columns(3),

                Section::make('Results Display')
                    ->description('GLOBAL settings that control results access. These CANNOT be overridden per-exam - they apply system-wide.')
                    ->schema([
                        Toggle::make('show_results_to_students')
                            ->label('Allow Results Page Access')
                            ->helperText('Allow students to access the detailed results page after exam submission'),

                        Toggle::make('show_score_immediately')
                            ->label('Show Score in Redirect Message')
                            ->helperText('If results page is disabled, show score in the redirect notification'),

                        Toggle::make('show_correct_answers_to_students')
                            ->label('Show Correct Answers')
                            ->helperText('Show which answers were correct after exam completion (requires results page access)'),
                    ])
                    ->columns(3),
            ])
            ->statePath('data');
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label('Save Settings')
                ->submit('save'),
        ];
    }

    public function save(): void
    {
        $data = $this->form->getState();
        
        $settings = app(ExamSettings::class);
        $settings->default_duration_minutes = (int) $data['default_duration_minutes'];
        $settings->default_passing_score = (int) $data['default_passing_score'];
        $settings->max_attempts = (int) $data['max_attempts'];
        $settings->shuffle_questions = (bool) $data['shuffle_questions'];
        $settings->shuffle_options = (bool) $data['shuffle_options'];
        $settings->allow_review = (bool) $data['allow_review'];
        $settings->show_score_immediately = (bool) $data['show_score_immediately'];
        $settings->show_results_to_students = (bool) $data['show_results_to_students'];
        $settings->show_correct_answers_to_students = (bool) $data['show_correct_answers_to_students'];
        $settings->save();

        Notification::make()
            ->title('Settings saved successfully')
            ->success()
            ->send();
    }
}
