<?php

namespace App\Filament\Pages;

use App\Settings\BrandingSettings;
use Filament\Actions\Action;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\FileUpload;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use BackedEnum;
use UnitEnum;

class ManageBrandingSettings extends Page
{
    protected static string|BackedEnum|null $navigationIcon = 'heroicon-o-paint-brush';

    protected string $view = 'filament.pages.manage-settings';

    protected static string|UnitEnum|null $navigationGroup = 'Settings';

    protected static ?int $navigationSort = 11;

    protected static ?string $title = 'Branding Settings';

    protected static ?string $navigationLabel = 'Branding';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(BrandingSettings::class);
        
        $this->form->fill([
            'logo_path' => $settings->logo_path ? [$settings->logo_path] : [],
            'favicon_path' => $settings->favicon_path ? [$settings->favicon_path] : [],
            'auth_background_image' => $settings->auth_background_image ? [$settings->auth_background_image] : [],
            'primary_color' => $settings->primary_color,
            'secondary_color' => $settings->secondary_color,
            'accent_color' => $settings->accent_color,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Logo & Identity')
                    ->description('Upload your organization logo and favicon.')
                    ->schema([
                        FileUpload::make('logo_path')
                            ->label('Logo')
                            ->image()
                            ->directory('branding')
                            ->disk('public')
                            ->visibility('public')
                            ->imageResizeMode('contain')
                            ->imageCropAspectRatio('1:1')
                            ->imageResizeTargetWidth('512')
                            ->imageResizeTargetHeight('512')
                            ->helperText('Recommended: PNG with transparent background, 512x512px or larger'),

                        FileUpload::make('favicon_path')
                            ->label('Favicon')
                            ->image()
                            ->directory('branding')
                            ->disk('public')
                            ->visibility('public')
                            ->helperText('Small icon shown in browser tabs (ICO or PNG, 32x32px recommended)'),

                        FileUpload::make('auth_background_image')
                            ->label('Login/Auth Background Image')
                            ->image()
                            ->directory('branding')
                            ->disk('public')
                            ->visibility('public')
                            ->imageResizeMode('cover')
                            ->imageResizeTargetWidth('1920')
                            ->imageResizeTargetHeight('1080')
                            ->helperText('Background image for login and authentication pages (1920x1080px recommended)'),
                    ])
                    ->columns(2),

                Section::make('Color Scheme')
                    ->description('Define your brand colors for the application.')
                    ->schema([
                        ColorPicker::make('primary_color')
                            ->label('Primary Color')
                            ->required()
                            ->helperText('Main brand color used for buttons and accents'),

                        ColorPicker::make('secondary_color')
                            ->label('Secondary Color')
                            ->required()
                            ->helperText('Secondary color for highlights'),

                        ColorPicker::make('accent_color')
                            ->label('Accent Color')
                            ->required()
                            ->helperText('Accent color for special elements'),
                    ])
                    ->columns(3),
            ])
            ->statePath('data');
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('save')
                ->label('Save Settings')
                ->submit('save'),
        ];
    }

    public function save(): void
    {
        $data = $this->form->getState();
        
        // FileUpload returns array, extract the path string
        $logoPath = $data['logo_path'] ?? null;
        $faviconPath = $data['favicon_path'] ?? null;
        $authBackgroundImage = $data['auth_background_image'] ?? null;
        
        // Handle array format from FileUpload
        if (is_array($logoPath)) {
            $logoPath = !empty($logoPath) ? reset($logoPath) : null;
        }
        if (is_array($faviconPath)) {
            $faviconPath = !empty($faviconPath) ? reset($faviconPath) : null;
        }
        if (is_array($authBackgroundImage)) {
            $authBackgroundImage = !empty($authBackgroundImage) ? reset($authBackgroundImage) : null;
        }
        
        $settings = app(BrandingSettings::class);
        $settings->logo_path = $logoPath;
        $settings->favicon_path = $faviconPath;
        $settings->auth_background_image = $authBackgroundImage;
        $settings->primary_color = $data['primary_color'];
        $settings->secondary_color = $data['secondary_color'];
        $settings->accent_color = $data['accent_color'];
        $settings->save();

        Notification::make()
            ->title('Settings saved successfully')
            ->success()
            ->send();
    }
}
